import json

from django.views.decorators.http import require_http_methods

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)

from ..controllers import phones as ctl
from ..forms.phones import PhonesGrid
from ..views.phones_view import phones as view_phones
from ..permissions.utils import observer_has_perm
from staff.lib.forms.errors import (
    invalid_json_error,
    default_error,
)

import logging
logger = logging.getLogger('person_profile.edit_views.edit_phones_view')

FORM_NAME = 'phones'


@responding_json
@available_for_external
@require_http_methods(['GET', 'POST'])
@observer_has_perm('edit_phones')
@use_request_lang
def edit_phones(request, login):
    if request.method == 'GET':
        phones_grid = PhonesGrid(
            initial=ctl.PhonesCtl(
                target_login=login,
                permissions_ctl=request.permissions_ctl,
            ).get_phones(
                for_view=False
            )
        )
        return {
            'choices': phones_grid.choices_as_front_dict(),
            FORM_NAME: phones_grid.as_front_dict(),
        }

    try:
        phones_data = json.loads(request.body).get(FORM_NAME, [])
    except (ValueError, AttributeError):
        logger.exception('Wrong JSON: %s', request.body)
        return invalid_json_error(request.body), 400

    grid = PhonesGrid(data=phones_data)

    if not grid.is_valid():
        return {'errors': {FORM_NAME: grid.errors()}}

    try:
        ctl.PhonesCtl(
            target_login=login,
            permissions_ctl=request.permissions_ctl,
        ).update_phones(grid.cleaned_data)
    except ctl.PhonesError:
        return default_error(), 500

    return view_phones(request, login)
