from django.core.exceptions import ValidationError

from staff.lib.forms.staff_form import StaffForm
from staff.lib.forms.staff_form_grid import StaffFormGrid
from staff.lib.forms.staff_fields import (
    StaffCharField,
    StaffIntegerField,
)

from staff.person.models import Bicycle

MAX_NUM = 10**8
MIN_NUM = 10**5


def check_ISO_7064_mod97(value):
    return value % 97 == 1


class BicycleGrid(StaffFormGrid):
    def __init__(self, form_class, initial=None, data=None, owner_login=None):
        self.owner_login = owner_login
        super(BicycleGrid, self).__init__(form_class, initial, data)

    def errors(self):
        if self._errors is not None:
            return self._errors
        self._errors = {}

        all_plates = [f.data['plate'] for f in self.forms]
        if len(set(all_plates)) != len(all_plates):
            self._errors.setdefault(self.grid_level_error_key, []).append(
                {'error_key': 'duplicate-plate'}
            )

        all_plates = [
            int(plate_no) for plate_no in all_plates
            if plate_no.isdigit()
        ]
        owned_plates = list(
            Bicycle.objects
            .filter(plate__in=all_plates)
            .exclude(owner__login=self.owner_login)
            .values_list('plate', flat=True)
        )

        for i, form in enumerate(self.forms):
            if not form.is_valid():
                self._errors[i] = form.errors
            if form.cleaned_data.get('plate') in owned_plates:
                self._errors[i] = {'plate': [{'error_key': 'plate-is-owned'}]}

        for attr_name in dir(self):
            if attr_name.startswith('validate_'):
                try:
                    getattr(self, attr_name)()
                except ValidationError as e:
                    self._errors.setdefault(
                        self.grid_level_error_key, []
                    ).append({'error_key': e.message})

        return self._errors


class BicycleForm(StaffForm):
    strip_fields = ['plate', 'description']

    id = StaffIntegerField(required=False, front_params={'hidden': True})
    plate = StaffIntegerField(required=True)
    description = StaffCharField(required=False, max_length=50)

    def clean_plate(self):
        plate_num = self.cleaned_data['plate']

        if not MIN_NUM <= plate_num <= MAX_NUM:
            raise ValidationError('{"error_key": "invalid"}', code='invalid')
        if not check_ISO_7064_mod97(plate_num):
            raise ValidationError('{"error_key": "invalid-mod97"}', code='invalid_mod97')
        return plate_num

    def clean_description(self):
        return self.cleaned_data['description']

    def clean(self):
        return super(BicycleForm, self).clean()
