import re

from django.core.exceptions import ValidationError

from staff.lib.forms.staff_form import StaffForm
from staff.lib.forms.staff_fields import StaffCharField, StaffIntegerField


_RE_W = re.compile(r'\s{2,}')
_RE_AD = re.compile(r'([A-ZА-ЯЁ]+)([0-9]+)')
_RE_DA = re.compile(r'([0-9]+)([A-ZА-ЯЁ]+)')
_RE_A = re.compile(r'^[0-9A-ZА-ЯЁ \-]+$')


def _prepare_plate(plate):
    plate = plate.strip()
    plate = re.sub(_RE_W, ' ', plate)
    plate = plate.upper()
    plate = re.sub(_RE_AD, r'\1 \2', plate)
    plate = re.sub(_RE_DA, r'\1 \2', plate)

    return plate


class CarPlateValidator(object):

    def __call__(self, value):
        if not _RE_A.match(_prepare_plate(value)):
            raise ValidationError('', code='invalid')


plate_validator = CarPlateValidator()


class CarForm(StaffForm):
    strip_fields = ['model', 'plate']

    id = StaffIntegerField(required=False, front_params={'hidden': True})
    model = StaffCharField(required=False, max_length=100)
    plate = StaffCharField(
        required=True,
        max_length=50,
        validators=[plate_validator],
    )

    def clean_plate(self):
        value = self.cleaned_data['plate']
        return value if not value else _prepare_plate(value.upper())
