import re

from django.core import validators

from staff.lib.forms.staff_form import StaffForm
from staff.lib.forms.staff_fields import (
    StaffBooleanField,
    StaffCharField,
    StaffIntegerField,
    StaffLocalizedModelChoiceField,
)

from staff.person.models import ContactType


class LoginValidator(validators.RegexValidator):
    regex = re.compile(r'^\s*[^\s/]+\s*$')
    message = '{"error_key": "staff-invalid_login_contact"}'
    code = 'staff_invalid_login_contact'


login_validator = LoginValidator()


class EmailValidator(validators.EmailValidator):
    message = '{"error_key": "staff-invalid_email_contact"}'
    code = 'staff_invalid_email_contact'


email_validator = EmailValidator()


class UrlValidator(validators.URLValidator):
    message = '{"error_key": "staff-invalid_url_contact"}'
    code = 'staff_invalid_url_contact'


url_validator = UrlValidator()


VALIDATORS = {
    'login': login_validator,
    'email': email_validator,
    'url': url_validator,
}


class ContactForm(StaffForm):
    strip_fields = ['account_id']

    id = StaffIntegerField(required=False, front_params={'hidden': True})
    contact_type = StaffLocalizedModelChoiceField(
        queryset=ContactType.objects.all().order_by('position'),
        required=True,
        empty_label=None,
    )
    account_id = StaffCharField(max_length=100, required=True)
    private = StaffBooleanField(initial=False, required=False)

    def clean_account_id(self):
        contact_type = self.cleaned_data.get('contact_type')
        if not contact_type:
            return
        account_id = self.cleaned_data.get('account_id')
        if contact_type.name_en == 'Telegram' and account_id.strip().startswith('@'):
            account_id = account_id.strip()[1:]

        VALIDATORS.get(contact_type.validation_type.lower())(account_id)
        return account_id
