from django.core.exceptions import ValidationError

from phonenumbers import (
    format_number,
    PhoneNumberFormat,
    parse,
    NumberParseException,
)

from staff.person.models import PHONE_KIND, PHONE_PROTOCOLS
from staff.lib.forms.staff_form_grid import StaffFormGrid

from staff.lib.forms.staff_form import StaffForm
from staff.lib.forms.staff_fields import (
    StaffBooleanField,
    StaffChoiceField,
    StaffCharField,
    StaffIntegerField,
    StaffPhoneField,
)


class PhoneForm(StaffForm):
    id = StaffIntegerField(required=False, front_params={'hidden': True})
    number = StaffPhoneField(max_length=50, required=True)
    protocol = StaffChoiceField(
        choices=PHONE_PROTOCOLS.choices(),
        required=True,
    )
    kind = StaffChoiceField(
        choices=PHONE_KIND.choices(),
        required=True,
    )
    description = StaffCharField(max_length=255, required=False)
    for_digital_sign = StaffBooleanField(required=False, front_params={'hidden': True})

    def clean_number(self):
        number = self.cleaned_data['number']
        try:
            number = format_number(parse(number), PhoneNumberFormat.INTERNATIONAL)
        except (NumberParseException, UnicodeDecodeError):
            raise ValidationError('{"error_key": "default-field-invalid"}', code='default-field-invalid')
        return number


class PhonesGrid(StaffFormGrid):
    form_class = PhoneForm

    def __init__(self, initial=None, data=None):
        super(PhonesGrid, self).__init__(
            form_class=self.form_class,
            initial=initial,
            data=data,
        )
