from typing import Optional

from staff.departments.models import Department, DepartmentRoles, DepartmentStaff
from staff.lib import waffle
from staff.lib.models.mptt import filter_by_heirarchy
from staff.person.models import Staff

from staff.person_profile.permissions.base import Properties


def can_view_digital_sign(_, perm, target_login):
    return (
        perm.is_superuser
        or
        perm.is_owner(target_login)
        or
        'django_intranet_stuff.can_view_digital_sign' in perm.permissions
    )


def can_view_phones(_, properties: Properties, target_login: Optional[str] = None) -> bool:
    if waffle.switch_is_active('hide_contacts'):
        has_perm = (
            properties.is_superuser
            or (target_login and properties.is_owner(target_login))
            or (target_login and properties.get_is_chief(target_login))
            or 'django_intranet_stuff.can_view_contacts' in properties.permissions
        )
        if not has_perm:
            return False
    return True


def can_view_phones_in_department(observer: Staff, department_url: str) -> bool:
    if waffle.switch_is_active('hide_contacts'):
        has_perm = observer.user.is_superuser or observer.user.has_perm('django_intranet_stuff.can_view_contacts')

        if has_perm:
            return True

        roles = DepartmentStaff.objects.filter(
            staff=observer,
            role__in=(
                DepartmentRoles.CHIEF.value,
                DepartmentRoles.HR_PARTNER.value,
                DepartmentRoles.GENERAL_DIRECTOR.value,
            ),
        )
        effective_roles = filter_by_heirarchy(
            roles,
            mptt_objects=[Department.objects.get(url=department_url)],
            by_children=False,
            include_self=True,
            filter_prefix='department__',
        )

        return effective_roles.exists()

    return True
