from typing import Dict, Any, Tuple, List

from staff.lib import waffle

from django.conf import settings
from django.contrib import auth
from django.db.models import Q

from staff.departments.models import DepartmentRoles
from staff.person.models import Staff, AFFILIATION

from staff.person_profile.constants import TARGET_TYPES


class Properties(object):
    required_fields = {
        'id',
        'login',
        'is_robot',
        'is_dismissed',
        'affiliation',
        'memorial_profile__intranet_status',
        'department_id',
        'department__id',
        'department__tree_id',
        'department__lft',
        'department__rght',
        'is_homeworker',
        'organization__country_code',
        'extra__staff_agreement',
    }

    def __init__(self, target_logins, observer, readonly, target_data=None):
        self.target_logins = target_logins
        self.observer = observer
        self.readonly = readonly

        self._targets_data = target_data
        self._observer_roles = None
        self._permissions = None

    def get_target_data(self, target_login: str) -> Dict[str, Any]:
        if self._targets_data is None:
            persons = (
                Staff.objects
                .values(*self.required_fields)
                .filter(login__in=self.target_logins)
            )
            if waffle.switch_is_active('rkn_mode'):
                persons = persons.filter(Q(is_dismissed=False) | Q(memorial_profile__intranet_status=1))
            self._targets_data = {p['login']: p for p in persons}
        return self._targets_data[target_login]

    def _get_observer_roles(self, roles: Tuple[str]) -> List[Dict]:
        if self._observer_roles is None:
            self._observer_roles = (
                self.observer.departmentstaff_set
                .values('role_id', 'department__tree_id', 'department__rght', 'department__lft')
            )
        return [role for role in self._observer_roles if role['role_id'] in roles]

    @property
    def permissions(self):
        if self._permissions is None:
            self._permissions = auth.get_backends()[0].get_all_permissions(self.observer.user)
        return self._permissions

    @staticmethod
    def _is_chief_for_target(observer_roles: List[Dict], target_data: Dict) -> bool:
        for role in observer_roles:
            if (
                role['department__tree_id'] == target_data['department__tree_id'] and
                role['department__lft'] <= target_data['department__lft'] and
                role['department__rght'] >= target_data['department__rght']
            ):
                return True
        return False

    def _check_roles(self, target_login: str, roles: Tuple) -> bool:
        if self.is_owner(target_login):
            return False

        return self._is_chief_for_target(
            observer_roles=self._get_observer_roles(roles),
            target_data=self.get_target_data(target_login)
        )

    def get_is_chief(self, target_login: str) -> bool:
        return self._check_roles(
            target_login=target_login,
            roles=(
                DepartmentRoles.CHIEF.value,
                DepartmentRoles.HR_PARTNER.value,
                DepartmentRoles.GENERAL_DIRECTOR.value,
            )
        )

    def get_is_hr_partner(self, target_login):
        return self._check_roles(
            target_login=target_login,
            roles=(DepartmentRoles.HR_PARTNER.value,),
        )

    def get_is_hr_analyst(self, target_login):
        return self._check_roles(
            target_login=target_login,
            roles=(DepartmentRoles.HR_ANALYST.value,),
        )

    def get_is_fincab_viewer(self, target_login):
        return self._check_roles(
            target_login=target_login,
            roles=(DepartmentRoles.FINCAB_VIEWER.value,),
        )

    def is_observer_from_yandex(self):
        return self.observer.affiliation == AFFILIATION.YANDEX

    def is_observer_from_yamoney(self):
        return self.observer.affiliation == AFFILIATION.YAMONEY

    def is_observer_from_external(self):
        return self.observer.affiliation == AFFILIATION.EXTERNAL

    def is_yandex(self, target_login):
        return self.get_target_data(target_login)['affiliation'] == AFFILIATION.YANDEX

    def is_external(self, target_login):
        return self.get_target_data(target_login)['affiliation'] == AFFILIATION.EXTERNAL

    def is_owner(self, target_login):
        return self.observer.login == target_login

    @property
    def is_superuser(self):
        return self.observer.user.is_superuser

    def is_agreed_dismissed(self, target_login):
        data = self.get_target_data(target_login)
        return data['is_dismissed'] and data['extra__staff_agreement']

    def is_dismissed(self, target_login):
        return self.get_target_data(target_login)['is_dismissed']

    def is_memorial(self, target_login):
        return self.get_target_data(target_login)['memorial_profile__intranet_status']

    def is_robot(self, target_login):
        return self.get_target_data(target_login)['is_robot']

    def is_country_enabled_digital_sign(self, target_login):
        data = self.get_target_data(target_login)
        return str(data['organization__country_code']).lower() in settings.DIGITAL_SIGN_COUNTRIES

    def is_homeworker(self, target_login):
        return bool(self.get_target_data(target_login)['is_homeworker'])

    def get_target_type(self, target_login):
        if self.is_memorial(target_login):
            return TARGET_TYPES.memorial
        elif self.is_agreed_dismissed(target_login):
            return TARGET_TYPES.agreed_dismissed
        elif self.is_dismissed(target_login):
            return TARGET_TYPES.dismissed
        elif self.is_robot(target_login):
            return TARGET_TYPES.robot
        else:
            return TARGET_TYPES.employee
