from staff.departments.models import Department
from staff.dismissal.objects import DismissalCtl
from staff.person.models import Staff, AFFILIATION

from staff.person_profile.constants import TARGET_TYPES
from staff.person_profile.permissions.base import Block, LoadableBlock, Link, Pencil, BaseRegistry
from staff.person_profile.permissions.check_permissions import can_view_digital_sign, can_view_phones


class BlockRegistry(BaseRegistry):
    # Blocks
    def _external_has_access_to_achievements(self, properties, target_login):
        return 'achievery.external_with_achievery_access' in properties.permissions

    achievements = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.agreed_dismissed,
        ),
        rule_for_external=_external_has_access_to_achievements,
        external_owner=True,
    )
    activity = LoadableBlock(
        show_on=(TARGET_TYPES.employee,)
    )
    goals = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )
    calendar = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )
    calendar_holidays = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )

    slack_status = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )

    def _external_has_access_to_self_gap(self, properties, target_login):
        return (
            properties.is_owner(target_login) and
            'django_intranet_stuff.can_view_gap_as_external' in properties.permissions
        )

    calendar_gaps = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        rule_for_external=_external_has_access_to_self_gap,
        external_other=True,
    )
    candidate_info = LoadableBlock(
        show_on=(TARGET_TYPES.employee,)
    )
    cars = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )
    bicycles = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )

    def _externals_has_access_to_contacts(self, properties, target_login):
        permission_name = 'django_intranet_stuff.can_view_contacts_as_external'
        return (
            properties.get_target_type(target_login) != TARGET_TYPES.dismissed
            and permission_name in properties.permissions
            and properties.observer.has_access_to_department_profiles(
                properties.get_target_data(target_login)['department_id'],
            )
        )

    contacts = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        rule_for_external=_externals_has_access_to_contacts,
        external_owner=True,
    )
    documents = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        owner=True,
        external_owner=True,
    )

    phones = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
        ),
        external_owner=True,
        custom_rule=can_view_phones,
    )
    survey = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        owner=True,
        external_owner=True,
    )
    responsible_for_robot = LoadableBlock(
        show_on=(TARGET_TYPES.robot,)
    )
    photos = LoadableBlock(
        show_on=TARGET_TYPES._choices,
        external_owner=True,
        external_other=True,
    )
    departments = LoadableBlock(
        show_on=TARGET_TYPES._choices,
        external_owner=True,
        external_other=True,
    )

    def _owner_in_yandex(self, properties, target_login):
        return properties.is_yandex(target_login)

    def _vs_has_umbrellas(self, properties, target_login: str) -> bool:
        if not properties.is_yandex(target_login):
            return False

        from staff.umbrellas.models import Umbrella
        from staff.budget_position.models import BudgetPositionAssignment

        assignment_qs = (
            BudgetPositionAssignment.objects.active()
            .filter(main_assignment=True, person__login=target_login)
            .values_list('value_stream_id', flat=True)
        )

        from staff.lib.models.mptt import filter_by_heirarchy
        umbrellas = filter_by_heirarchy(
            Umbrella.objects.active(),
            Department.valuestreams.filter(id__in=assignment_qs),
            filter_prefix='value_stream__',
            by_children=False,
            include_self=True,
        )

        return umbrellas.exists()

    def _can_edit_umbrellas(self, properties, target_login: str) -> bool:
        if not properties.is_yandex(target_login):
            return False

        return properties.get_is_chief(target_login) and BlockRegistry._vs_has_umbrellas(self, properties, target_login)

    value_streams = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
        custom_rule=_owner_in_yandex,
    )

    umbrellas = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
        custom_rule=_vs_has_umbrellas,
    )

    is_calendar_vertical = LoadableBlock(
        show_on=TARGET_TYPES._choices,
        external_owner=True,
        external_other=True,
    )
    birthday = Block(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        external_owner=True,
    )
    services = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
            TARGET_TYPES.dismissed,
        ),
        external_owner=True,
    )
    table_books = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )
    hardware = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=True,
        owner=True,
        permission='django_intranet_stuff.can_view_hardware',
        waffle='hide_hardware_block',
        external_owner=True,
    )
    software = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=True,
        owner=True,
        permission='django_intranet_stuff.can_view_software',
        waffle='hide_software_block',
        external_owner=True,
    )
    edoc = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        owner=True,
        permission='django_intranet_stuff.can_view_meals_sum',
        waffle='hide_food_block',
        external_owner=True,
    )
    digital_sign_status = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        owner=True,
        external_owner=True,
    )
    digital_sign_certification_status = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        custom_rule=can_view_digital_sign,
        waffle='hide_digital_sign_certification_status',
    )
    approvements = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        owner=True,
        waffle='hide_approvements',
    )
    password = Block(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot),
        owner=True,
        permission='django_intranet_stuff.can_view_password_block',
        waffle='hide_password_block',
        external_owner=True,
    )
    vacation = Block(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
        permission='django_intranet_stuff.can_view_vacation',
        waffle='hide_vacation_block',
        external_owner=True,
    )
    paid_day_off = Block(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
        permission='django_intranet_stuff.can_view_paid_day_off',
        waffle='hide_paid_day_off_block',
        external_owner=True,
    )
    location_office = Block(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.agreed_dismissed,
        ),
        external_owner=True,
    )
    location_room = Block(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )
    location_table = Block(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )
    work_mode = Block(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )
    memorial_info = Block(
        show_on=(TARGET_TYPES.memorial,)
    )
    chief = LoadableBlock(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot),
        external_owner=True,
        external_other=True,
    )
    value_stream_chief = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
        custom_rule=_owner_in_yandex,
        waffle='hide_value_stream_chief_block',
    )
    hr_partners = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )
    curators = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )
    duties = Block(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
            TARGET_TYPES.dismissed,
        ),
        external_owner=True,
        external_other=True,
    )

    def _can_view_official_base(self, properties, target_login):
        permission_name = 'django_intranet_stuff.can_view_dismissed_official_base'
        return (
            properties.get_target_type(target_login) != TARGET_TYPES.dismissed
            or permission_name in properties.permissions
        )

    official_base = Block(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
            TARGET_TYPES.dismissed,
        ),
        custom_rule=_can_view_official_base,
    )
    fio = Block(
        show_on=TARGET_TYPES._choices,
        custom_rule=_can_view_official_base,
        external_owner=True,
        external_other=True,
    )
    work_phone = Block(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
        ),
        external_owner=True,
        external_other=True,
    )
    official_organization = Block(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )

    def _can_view_quit_at(self, properties, target_login):
        permission_name = 'django_intranet_stuff.can_view_quit_at'
        return (
            properties.get_target_type(target_login) == TARGET_TYPES.agreed_dismissed
            or permission_name in properties.permissions
        )

    quit_at = Block(
        show_on=(
            TARGET_TYPES.agreed_dismissed,
            TARGET_TYPES.dismissed,
        ),
        custom_rule=_can_view_quit_at,
    )
    quit_at_short = Block(
        show_on=(
            TARGET_TYPES.dismissed,
        ),
    )
    external_login = Block(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.agreed_dismissed,
            TARGET_TYPES.robot,
        ),
        external_owner=True,
    )
    ssh_keys = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        external_owner=True,
    )
    gpg_keys = LoadableBlock(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        external_owner=True,
    )
    education = Block(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )
    personal = Block(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
    )

    digital_sign = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        custom_rule=can_view_digital_sign,
    )

    def _can_view_occupation(self, properties, target_login):
        return properties.get_is_hr_analyst(target_login) or properties.get_is_hr_partner(target_login)

    occupation = LoadableBlock(
        show_on=(TARGET_TYPES.employee,),
        custom_rule=_can_view_occupation,
    )

    # Links
    salary = Link(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        permission='django_intranet_stuff.can_view_anketa_wish_form_button',
    )

    feedback = Link(
        show_on=(TARGET_TYPES.employee,)
    )

    dismiss_create = Link(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot, TARGET_TYPES.memorial),
        chief=True,
        permission='dismissal.can_dismiss_from_anketa',
        waffle='hide_dismiss_link',
        not_readonly=True,
    )

    def _dismissal_edit_rule(self, perm, target_login):
        target_id = perm.get_target_data(target_login)['id']
        dismissal = DismissalCtl().get(target_id)
        return dismissal is not None and DismissalCtl().can_see_dismissal_link(
            object_=Staff.objects.get(id=target_id),
            subject=perm.observer,
        )

    dismiss_edit = Link(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot, TARGET_TYPES.memorial),
        chief=True,
        waffle='hide_dismiss_link',
        not_readonly=True,
        custom_rule=_dismissal_edit_rule,
    )

    order_card = Link(
        show_on=(TARGET_TYPES.employee,),
        owner=True,
        waffle='hide_order_card_link'
    )

    revoke_ssh = Link(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=False,
        owner=False,
        permission='django_intranet_stuff.can_withdraw_keys',
        waffle='hide_ssh_link',
        not_readonly=True,
    )

    add_external_login = Link(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot),
        owner=True,
        not_readonly=True,
        custom_rule=lambda self, perm, target_login: (
            perm.get_is_chief(target_login) and 'django_intranet_stuff.can_manage_external_logins' in perm.permissions
        )
    )
    delete_external_login = add_external_login

    upravlyator = Link(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot),
        chief=True,
        owner=True,
        permission='django_intranet_stuff.can_view_upravlyator_link',
    )

    def _cabinet_link_show_rule(self, properties, target_login):
        PERMISSION_NAME = 'django_intranet_stuff.can_view_cabinet_link'
        return (
            properties.get_is_chief(target_login)
            or properties.is_owner(target_login)
        ) and PERMISSION_NAME in properties.permissions

    cabinet = Link(
        show_on=(TARGET_TYPES.employee,),
        custom_rule=_cabinet_link_show_rule,
    )

    ml = Link(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
    )

    yamb = Link(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )

    slack = Link(
        show_on=(TARGET_TYPES.employee,),
        external_owner=True,
        external_other=True,
    )

    finances = Link(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
    )

    def _can_view_fincab(self, properties, target_login):
        can_view = (
            properties.get_is_hr_analyst(target_login)
            or properties.get_is_hr_partner(target_login)
            or properties.get_is_fincab_viewer(target_login)
        )
        return can_view

    fincab = Link(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
        custom_rule=_can_view_fincab,
        waffle='hide_fincab_block',
    )

    delete_all_photos = Link(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot),
        chief=True,
        owner=True,
        not_readonly=True,
        permission='django_intranet_stuff.can_delete_all_photos',
        external_owner=True,
    )

    # Pencils
    upload_photo = Pencil(
        show_on=(TARGET_TYPES.employee, TARGET_TYPES.robot),
        chief=True,
        owner=True,
        not_readonly=True,
        permission='django_intranet_stuff.can_upload_photo',
        external_owner=True,
    )
    edit_head = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=True,
        owner=True,
        permission='django_intranet_stuff.can_edit_head',
        not_readonly=True,
        external_owner=True,
    )
    edit_duties = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=True,
        owner=True,
        not_readonly=True,
        external_owner=True,
    )
    edit_digital_sign = Pencil(
        show_on=(
            TARGET_TYPES.employee,
        ),
        chief=False,
        owner=False,
        custom_rule=lambda self, props, login: (
            self.check_owner(props, login) and
            props.is_country_enabled_digital_sign(login)
        ),
        not_readonly=True,
        external_owner=True,
    )
    edit_documents = Pencil(
        show_on=(
            TARGET_TYPES.employee,
        ),
        chief=False,
        owner=True,
        not_readonly=True,
        external_owner=True,
    )
    edit_location = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=True,
        owner=True,
        permission='django_intranet_stuff.can_edit_location',
        not_readonly=True,
        external_owner=True,
    )
    edit_phones = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=True,
        owner=True,
        not_readonly=True,
        external_owner=True,
    )
    edit_contacts = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        chief=True,
        owner=True,
        not_readonly=True,
        external_owner=True,
    )
    edit_cars = Pencil(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
        not_readonly=True,
        external_owner=True,
    )
    edit_bicycles = Pencil(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
        not_readonly=True,
        external_owner=True,
    )
    edit_candidate = Pencil(
        show_on=(TARGET_TYPES.employee,),
        owner=True,
        not_readonly=True,
    )
    edit_other = Pencil(
        show_on=(TARGET_TYPES.employee,),
        chief=True,
        owner=True,
        not_readonly=True,
    )
    edit_department = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        permission='django_intranet_stuff.can_edit_department',
        not_readonly=True,
    )
    edit_umbrellas = Pencil(
        show_on=(TARGET_TYPES.employee,),
        chief=False,
        owner=False,
        custom_rule=_can_edit_umbrellas,
    )
    edit_official = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.agreed_dismissed,
        ),
        permission='django_intranet_stuff.can_edit_official',
        not_readonly=True,
    )
    edit_settings = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.memorial,
            TARGET_TYPES.agreed_dismissed,
        ),
        owner=True,
        not_readonly=True,
        external_owner=True,
    )

    edit_gpg_keys = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
        ),
        owner=True,
        permission='django_intranet_stuff.can_edit_keys',
        not_readonly=True,
        external_owner=True,
    )

    def _external_ssh_key_edit_rule(self, properties, target_login):
        PERMISSION_NAME = 'django_intranet_stuff.can_manage_ssh_key_for_externals'
        return (
            properties.get_is_chief(target_login) and
            PERMISSION_NAME in properties.permissions and
            properties.is_external(target_login) and
            properties.observer.affiliation != AFFILIATION.EXTERNAL
        )

    edit_ssh_keys = Pencil(
        show_on=(
            TARGET_TYPES.employee,
            TARGET_TYPES.robot,
            TARGET_TYPES.agreed_dismissed,
        ),
        owner=True,
        permission='django_intranet_stuff.can_edit_keys',
        not_readonly=True,
        custom_rule=_external_ssh_key_edit_rule,
        external_owner=True,
    )
