import logging
from typing import Optional, List

from staff.celery_app import app
from staff.lib.tasks import LockedTask
from staff.lib.startrek.issues import startrek_issues_repository
from staff.person.models.person import WORK_MODES
from staff.users.models import Staff
from staff.person.controllers import PersonCtl

from startrek_client.objects import Resource


logger = logging.getLogger('staff.person_profile.tasks')


RESULT_MAPPING = {
    'Я буду постоянно в офисе (4 или 5 дней в неделю)': WORK_MODES.OFFICE,
    'Я буду работать в офисе 3 дня в неделю': WORK_MODES.MIXED,
    'Я буду работать в офисе 2 дня в неделю': WORK_MODES.MIXED,
    'Я буду работать в офисе 1 день в неделю': WORK_MODES.MIXED,
    'Я буду почти всегда работать из дома (редко в офисе)': WORK_MODES.REMOTE,
    'I will be working from the office all the time (4 or 5 days per week)': WORK_MODES.OFFICE,
    'I will be working from the office 2 days per week': WORK_MODES.MIXED,
    'I will be working from the office 3 days per week': WORK_MODES.MIXED,
    'I will be working from the office 1 day per week': WORK_MODES.MIXED,
    'I will be mostly working from home (rarely from the office)': WORK_MODES.REMOTE,
}


def get_issues(count_days: int = None) -> List[Resource]:
    payload = {
        'query': 'Queue: OFFICEMODE and status: confirmed and result: notEmpty()',
    }
    if count_days:
        payload['query'] += f' and updated: >= today() - {count_days}d'

    return startrek_issues_repository.get(lookup=payload)


def get_new_work_mode(issue: Resource) -> Optional[str]:
    result = issue.result
    return RESULT_MAPPING.get(result, None)


@app.task(ignore_result=True)
class UpdateWorkMode(LockedTask):
    def locked_run(self, *args, **kwargs):
        update_work_modes(count_days=2)


def update_work_mode(issue: Resource) -> int:
    work_mode = get_new_work_mode(issue)
    if not work_mode:
        logger.warning('Unknown result in %s', issue['key'])
        return 0

    try:
        login = issue.employee.id
    except AttributeError:
        logger.warning('Unknown login in %s', issue['key'])
        return 0

    try:
        staff = Staff.objects.get(login=login)
    except Staff.DoesNotExist:
        logger.warning('User %s not found. Skipped.', login)
        return 0

    person_ctl = PersonCtl(staff)
    if staff.work_mode != work_mode:
        person_ctl.update({'work_mode': work_mode}).save()
        return 1

    return 0


def update_work_modes(count_days: int = None) -> None:
    issues_count, updated_count = 0, 0
    issues = get_issues(count_days)
    for issue in issues:
        issues_count += 1
        updated_count += update_work_mode(issue)
    logger.info('Total: %s issues. %s profiles updated work_mode.', issues_count, updated_count)
