from django.db.models import Q
from django.views.decorators.http import require_GET
from django.utils.translation import get_language

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.achievery.context import AchievementsContext
from staff.achievery.models import Icon

from staff.person_profile.permissions.utils import observer_has_perm


@require_GET
@available_for_external
@responding_json
@observer_has_perm('achievements')
@use_request_lang
def achievements(request, login):
    observer = request.user.get_profile()
    lang = get_language()

    given_achievements = [
        {
            'achievement_id': given.achievement.id,
            'level': given.level,
            'title': getattr(given.achievement.title, lang),
        } for given in AchievementsContext(login, observer)
    ]

    if given_achievements:
        icons = _get_icons(given_achievements)

        result = [
            {
                'achievement_id': given['achievement_id'],
                'icon_id': icons.get((given['achievement_id'], given['level'])),
                'title': given['title'],
            } for given in given_achievements
        ]

        return {'target': {'achievements': result}}

    return {}


def _get_icons(given_achievements):
    icon_lookup = Q(pk=None)
    for given in given_achievements:
        icon_lookup |= Q(achievement_id=given['achievement_id'], level=given['level'])
    icons = {
        (icon['achievement_id'], icon['level']): icon['id']
        for icon in Icon.objects.filter(icon_lookup).values('achievement_id', 'level', 'id')
    }
    return icons
