import json
import logging

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import AnonymousUser
from django.views.decorators.http import require_GET

from staff.gap.api.views.need_for_attention_views import _get_unconfirmed_gaps
from staff.gap.controllers.unconfirmed import _subordinates_ids
from staff.lib import requests, tvm2
from staff.lib.decorators import responding_json, auth_by_tvm_only
from staff.person_profile.controllers.subordinates import direct_subordinates_q
from staff.person_profile.permissions.utils import observer_has_perm
from staff.preprofile.repository import Repository


logger = logging.getLogger(__name__)


def get_ok_approvements_count(request, login):
    url = f'https://{settings.OK_HOST}/_api/staff/approvements_counts/'
    user_ticket = request.yauser.raw_user_ticket
    service_ticket = tvm2.get_tvm_ticket_by_deploy('ok')

    try:
        response = requests.get(
            url=url,
            headers={
                tvm2.TVM_USER_TICKET_HEADER: user_ticket,
                tvm2.TVM_SERVICE_TICKET_HEADER: service_ticket,
            },
            timeout=(1, 3, 5),
        )
        response.raise_for_status()
        counts = response.json()
    except requests.RequestException:
        logger.warning(
            'Request for approvements count of %s@ failed',
            login,
            exc_info=True,
        )
        counts = {}
    except json.JSONDecodeError:
        logger.warning('Invalid answer from ok', exc_info=True)
        counts = {}
    return counts.get('current_count', 0)


def get_gap_approvements_count(person_id):
    direct_subordinates_ids = _subordinates_ids(direct_subordinates_q(person_id))
    unconfirmed_gaps = _get_unconfirmed_gaps(direct_subordinates_ids)
    return unconfirmed_gaps.count()


def get_newhire_approvements_count(person):
    preprofiles_count = Repository(person).preprofiles_waiting_direct_chief_to_confirm().count()
    return preprofiles_count


@login_required
@responding_json
@require_GET
@observer_has_perm('approvements')
def approvements_count(request, login):
    person = request.user.get_profile()

    ok_count = get_ok_approvements_count(request, login)
    gap_count = get_gap_approvements_count(person.id)
    newhire_count = get_newhire_approvements_count(person)

    return {
        'target': {
            'approvements': {
                'ok': ok_count,
                'gap': gap_count,
                'newhire': newhire_count,
            },
        },
    }


@responding_json
@require_GET
@auth_by_tvm_only(['ok'])
def gap_newhire_counts(request):
    if isinstance(request.user, AnonymousUser):
        return {'error': 'user ticket is required'}, 400

    person = request.user.get_profile()
    gap_count = get_gap_approvements_count(person.id)
    newhire_count = get_newhire_approvements_count(person)

    return {
        'gap': gap_count,
        'newhire': newhire_count,
    }
