from typing import Dict, Type

from django.views.decorators.http import require_http_methods

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.exceptions import ErrorWithStatusCode
from staff.lib.utils.qs_values import localize
from staff.person_profile.controllers.blocks.abstract_block import AbstractBlock
from staff.person_profile.controllers.blocks.chief_block import ChiefBlock
from staff.person_profile.controllers.blocks.department_block import DepartmentBlock
from staff.person_profile.controllers.blocks.hr_partner_block import HRPartnerBlock
from staff.person_profile.controllers.blocks.curators_block import CuratorsBlock
from staff.person_profile.controllers.blocks.occupation_block import OccupationBlock
from staff.person_profile.controllers.blocks.umbrellas_block import UmbrellasBlock
from staff.person_profile.controllers.blocks.value_stream_block import ValueStreamBlock
from staff.person_profile.controllers.blocks.vaue_stream_chief_block import ValueStreamChiefBlock
from staff.person_profile.controllers.search import get_person
from staff.person_profile.permissions.properties import Properties
from staff.person_profile.permissions.registry import BlockRegistry
from staff.person_profile.permissions.utils import exception_views, view403


def _get_block_data(request, login: str, perm_name: str, block_cls: Type[AbstractBlock]) -> Dict:
    target_data = localize(get_person(login, set(block_cls.fields) | Properties.required_fields))

    properties = Properties(
        target_logins=[login],
        observer=request.user.get_profile(),
        readonly=request.service_is_readonly,
        target_data={login: target_data},
    )
    block_registry = BlockRegistry(properties)

    try:
        has_perm = block_registry.is_available(perm_name, login)
    except ErrorWithStatusCode as e:
        exception_view = exception_views.get(e.status_code)
        if exception_view:
            return exception_view(request, login)
        raise

    if not has_perm:
        return view403(request, login)

    block = block_cls(target_data, {})
    return {'target': block.get_data()}


@available_for_external
@responding_json
@require_http_methods(['GET', 'POST'])
@use_request_lang
def chief(request, login):
    return _get_block_data(request, login, 'chief', ChiefBlock)


@available_for_external
@responding_json
@require_http_methods(['GET'])
@use_request_lang
def value_stream_chief(request, login):
    return _get_block_data(request, login, 'value_stream_chief', ValueStreamChiefBlock)


@available_for_external
@responding_json
@require_http_methods(['GET', 'POST'])
@use_request_lang
def hr_partners(request, login):
    return _get_block_data(request, login, 'hr_partners', HRPartnerBlock)


@available_for_external
@responding_json
@require_http_methods(['GET', 'POST'])
@use_request_lang
def curators(request, login):
    return _get_block_data(request, login, 'curators', CuratorsBlock)


@available_for_external
@responding_json
@require_http_methods(['GET', 'POST'])
@use_request_lang
def departments(request, login):
    return _get_block_data(request, login, 'departments', DepartmentBlock)


@available_for_external
@responding_json
@require_http_methods(['GET', 'POST'])
@use_request_lang
def value_streams(request, login):
    return _get_block_data(request, login, 'value_streams', ValueStreamBlock)


@available_for_external
@responding_json
@require_http_methods(['GET'])
@use_request_lang
def umbrellas(request, login):
    return _get_block_data(request, login, 'umbrellas', UmbrellasBlock)


@available_for_external
@responding_json
@require_http_methods(['GET'])
@use_request_lang
def occupation(request, login):
    return _get_block_data(request, login, 'occupation', OccupationBlock)
