import logging
from datetime import datetime, timedelta
from dateutil import parser
from pytz import timezone, UTC

from django.views.decorators.http import require_GET
from django.conf import settings

from staff.gap.events_binder import EventsBinder
from staff.person.models import Staff
from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.requests import ConnectionError

from staff.person_profile.permissions.utils import observer_has_perm
from staff.person_profile.controllers.calendar import (
    fetch_events_from_calendar,
    hydrate_events,
    CalendarError,
    PersonHolidays,
)
from staff.person_profile.controllers.gap import get_calendar_gaps


logger = logging.getLogger('staff.person_profile.views')


CALENDAR_PERIOD = timedelta(days=7)
GAPS_PERIOD = timedelta(weeks=6)


def _get_date_params(request):
    date_from_p = request.GET.get('date_from')
    date_to_p = request.GET.get('date_to')

    date_from = parser.parse(date_from_p).astimezone(UTC) if date_from_p else None
    date_to = parser.parse(date_to_p).astimezone(UTC) if date_to_p else None

    return date_from, date_to


@available_for_external
@responding_json
@require_GET
@observer_has_perm('calendar')
@use_request_lang
def calendar(request, login):
    observer = request.user.get_profile()

    if observer.login == login:
        target = observer
    else:
        target = Staff.objects.filter(login=login).first()

    if target is None:
        return {'error': 'Person with login %s not found' % login}, 404

    date_from, date_to = _get_date_params(request)

    o_tzinfo = timezone(observer.tz)
    t_tzinfo = timezone(target.tz)
    now = timezone(settings.TIME_ZONE).localize(datetime.now()).astimezone(o_tzinfo)

    date_from: datetime.date = date_from or t_tzinfo.localize(datetime(now.year, now.month, now.day))
    date_to: datetime.date = date_to or date_from + CALENDAR_PERIOD

    raw_user_ticket = request.yauser.raw_user_ticket

    try:
        events = fetch_events_from_calendar(
            observer_uid=observer.uid,
            observer_tvm_ticket=raw_user_ticket,
            target_email=target.work_email,
            date_from=date_from,
            date_to=date_to,
        )
    except (CalendarError, ConnectionError):
        return {}, 434

    events = hydrate_events(events)

    return {
        'target': {
            'events': list(events),
            'name': target.inflections.genitive.split(' ')[0],
        },
    }


@available_for_external
@responding_json
@require_GET
@observer_has_perm('calendar_gaps')
@use_request_lang
def calendar_gaps(request, login):
    observer = request.user
    try:
        target = Staff.objects.get(login=login)
    except Staff.DoesNotExist:
        return {'error': 'Person with login %s not found' % login}, 404

    res = {
        'target': {
            'calendar_gaps': [],
            'name': target.inflections.genitive.split(' ')[0],
        },
    }
    # STAFF-17109
    if login == settings.LOGIN_TIGRAN and observer.get_profile().login != settings.LOGIN_TIGRAN:
        return res

    date_from, date_to = _get_date_params(request)

    date_from = date_from or datetime.utcnow().replace(hour=0, minute=0, second=0)
    date_to = date_to or date_from + GAPS_PERIOD

    binder = EventsBinder(start_key='from', end_key='to')
    gaps = []
    for gap in get_calendar_gaps(date_from, date_to, login, observer):
        binder.bind(gap)
        gaps.append(gap)

    res['target']['calendar_gaps'] = gaps
    return res


@available_for_external
@responding_json
@require_GET
@observer_has_perm('calendar_holidays')
@use_request_lang
def calendar_holidays(request, login):
    try:
        target = Staff.objects.get(login=login)
    except Staff.DoesNotExist:
        return {'error': 'Person with login %s not found' % login}, 404

    date_from, date_to = _get_date_params(request)

    first_day = date_from and date_from.date() or datetime.utcnow().date()
    last_day = date_to and date_to.date() or first_day + CALENDAR_PERIOD

    try:
        holidays = PersonHolidays(person=target, first_day=first_day, last_day=last_day).get_as_dict()
    except (CalendarError, ConnectionError):
        logger.warning('Error getting holidays for %s', login)
        return {'error': 'Calendar is not responding.'}, 434

    return {
        'target': {
            'holidays': holidays,
        },
    }
