import re
from typing import List, Dict, AnyStr, Any, Generator

from django.views.generic.base import View
from django.utils.translation import ugettext as _
from django.http import Http404

from staff.lib.decorators import available_by_center_token

from staff.lib.decorators import responding_json, available_for_external

from staff.departments.models import Department
from staff.person.models import Staff
from staff.person_profile.permissions.properties import Properties
from staff.person_profile.views.cards.base import get_joined_persons_data
from staff.person_profile.cards.blocks_registry import CardsBlockRegistry
from staff.person_profile.cards.meta_blocks_registry import CardsMetaBlocksRegistry


PersonData = Dict[AnyStr, Any]


def _fix_undefined(card_data):
    if 'position' not in card_data:
        card_data['position'] = ''
    if 'last_office' not in card_data:
        card_data['last_office'] = {'ago': '', 'office': ''}
    if 'office' not in card_data:
        card_data['office'] = ''


def _clean_dismissed(card_data):
    if card_data.get('is_dismissed'):
        card_data['position'] = ''
        card_data.pop('phones', None)


def filter_persons_by_access(observer, persons_data):  # type: (Staff, List[PersonData]) -> Generator[PersonData]
    profiles_filter_q = observer.departments_by_outstaff_perm_query('django_intranet_stuff.can_view_profiles')

    permitted_department_ids = set(
        Department.objects
        .filter(profiles_filter_q)
        .values_list('id', flat=True)
    )

    for item in persons_data:
        if item['department']['id'] in permitted_department_ids:
            yield item


@available_by_center_token
@available_for_external
class StaffCards(View):

    def dispatch(self, request, *args, **kwargs):
        is_jsonp = request.GET.get('format', 'jsonp') == 'jsonp'

        if is_jsonp:
            request.GET._mutable = True
            request.GET['jsonp'] = request.GET.get('_callback', 'callback')
            request.GET._mutable = False

        return responding_json(
            view=super(StaffCards, self).dispatch,
            is_jsonp=is_jsonp,
        )(request, *args, **kwargs)

    def get_block_registry(self, observer_person, target_logins, readonly):
        properties = Properties(
            observer=observer_person,
            target_logins=target_logins,
            readonly=readonly,
        )
        return CardsBlockRegistry(properties)

    def get(self, request, *args, **kwargs):
        logins = self._get_logins()
        if not logins:
            return {}

        need_messages = self.request.GET.get('_messages')

        result = {}
        observer = request.user.get_profile()
        persons_data = get_joined_persons_data({'login__in': logins}, observer)

        if not observer.is_internal():
            persons_data = list(filter_persons_by_access(observer, persons_data))

        block_registry = self.get_block_registry(
            observer_person=observer,
            target_logins=[p['login'] for p in persons_data],
            readonly=request.service_is_readonly
        )

        for person_data in persons_data:
            meta_block_registry = CardsMetaBlocksRegistry(person_data, registry=block_registry)
            person_card_data = meta_block_registry.get_data()

            _fix_undefined(person_card_data)
            _clean_dismissed(person_card_data)

            result[person_data['login']] = person_card_data

        if need_messages:
            result.update(self.messages())

        return result

    _filter_logins_re = re.compile(r'[^a-z0-9\.\-_@]', re.IGNORECASE)

    @classmethod
    def _filter_logins(cls, logins):
        """
        Defense from evil requests with cyrillic and another bad symbols
        >>> StaffCards._filter_logins(['killa', u'горилла42', 'GodZilla'])
        ['killa', 'GodZilla']
        """
        return [
            login.lower() for login in logins
            if login
            and login not in ('format', 'jsonp')
            and not login.startswith('_')
            and not cls._filter_logins_re.search(login)
        ]

    def _get_logins(self):
        logins = self._filter_logins(self.request.GET.keys())
        if not logins:
            raise Http404()
        return logins

    def messages(self):
        return {'_messages': {
            'write_email': _('api.StaffCards_WriteEmail'),
            'write_jabber': _('api.StaffCards_WriteJabberMessage'),
            'number_will_be_dialed_up': _('api.StaffCards_NumberWillBeDialedUp'),
            'call': _('api.StaffCards_Call'),
            'cancel': _('api.StaffCards_Cancel'),
            'pick_up_receiver': _('api.StaffCards_PickUpReceiver'),
            'error_code': _('api.StaffCards_ErrorCode'),
        }}


staff_cards = StaffCards.as_view()
