import re

from django.http import Http404
from django.utils.translation import ugettext_lazy as _, string_concat
from django.conf import settings

from staff.lib.decorators import available_by_center_token

from staff.lib.decorators import responding_jsonp
from staff.apicenter.views import BaseCenterView
from staff.person_profile.cards.utils import format_date_ago

from .base import get_joined_persons_data


@available_by_center_token
class UserCardsView(BaseCenterView):

    clean_re = re.compile(r'[^A-Za-z0-9\.\-_@| ,]')

    def _get_logins(self):
        logins_str = self.request.GET.get('logins')
        if not logins_str:
            raise Http404
        # defense from evil request by at.yandex-team.ru
        logins_str = self.clean_re.sub('', logins_str)

        logins_or_emails = [l.strip() for l in logins_str.split(',')]

        return logins_or_emails, '@' in logins_str

    def __call__(self, request):
        return responding_jsonp(self.execute)(request)

    def execute(self, request):
        self.request = request
        self.request.GET._mutable = True
        self.request.GET['jsonp'] = 'document.PersonCard.onload'
        self.request.GET._mutable = False

        logins_or_emails, is_emails = self._get_logins()

        if not logins_or_emails:
            return {}

        if is_emails:
            query = {'work_email__in': logins_or_emails}
        else:
            query = {'login__in': logins_or_emails}

        observer = request.user.get_profile()
        persons = get_joined_persons_data(query, observer)

        result = {}
        key = 'work_email' if is_emails else 'login'

        for person in persons:
            item = self.extract_base(person)
            item.update(self.extract_userpic(person))
            item.update(self.extract_phones(person))
            item.update(self.extract_department(person))
            item.update(self.extract_direction(person))
            item.update(self.accounts(person))
            item.update(self.extract_last_office(person))
            item.update(self.extract_gap(person))

            result[person[key]] = item

        error = {'_error': 'User not found'}
        for login_or_email in logins_or_emails:
            if login_or_email not in result:
                result[login_or_email] = error

        return result

    def extract_base(self, person):
        fields = [
            'first_name',
            'last_name',
            'position',
            'work_email',
            'custom_work_email',
            'is_dismissed',
            'gender'
        ]
        fields = self._get_fields(fields)
        fields.append('login')

        base = {f: person[f] for f in fields}
        base['login_ld'] = base['login']
        return base

    def extract_userpic(self, person):
        url = '//center.yandex-team.ru/api/v1/user/%s/photo/100/square.jpg'
        return {'userpic': url % person['login']}

    def extract_phones(self, person):
        return {'phones': {
            'work': person['work_phone'],
            'mobile': person['mobile_phone'],
        }}

    def extract_department(self, person):
        if not person['department']:
            return {}

        dep = person['department']

        return {'group': {
            'css': dep['code'],
            'name': dep['short_name'] or dep['name'],
            'url': 0,
        }}

    def extract_direction(self, person):
        if not person['direction']:
            return {}

        dep = person['direction']

        return {'dep': {
            'css': dep['code'],
            'name': dep['short_name'] or dep['name'],
            'url': 0,
        }}

    def accounts(self, person):
        return {'services': [
            {
                'name': _('staff.base.title'),
                'url': '//%s/%s' % (settings.STAFF_HOST, person['login'])
            },
            {
                'name': _('api.StaffCards_Wiki'),
                'url': '//%s/~%s' % (settings.WIKI_HOST, person['login'])
            },
            {
                'name': _('api.StaffCards_Diary'),
                'url': '//%s.%s' % (person['login'], settings.ATUSHKA_HOST)
            },
            {
                'name': _('api.StaffCards_Meetings'),
                'url': '//%s/invite/my/%s' % (settings.CALENDAR_HOST, person['login'])
            },
        ]}

    def extract_last_office(self, person):
        last_activity = person['last_activity']
        if not last_activity:
            return {}

        return {'last_office': {
            'ago': format_date_ago(last_activity['updated_at'], None, False),
            'short_name': last_activity['office'],
        }}

    def extract_gap(self, person):
        gap = person['gap']
        if not gap:
            return {}

        _gaps = {
            'absence': _('is absent'),
            'duty': _('is duty'),
            'vacation': _('is on vacation'),
            'paid_day_off': _('is on paid day off'),
            'selfpaid': _('is on vacation'),  # oioioi hack!
            'illness': _('is ill'),
            'illness_covid': _('is ill covid'),
            'trip': _('is away on business'),
            'maternity_leave': _('is on parental leave'),
            'remote_work': _('is working remotely'),
            'office_work': _('is working in office'),
        }

        name = _gaps.get(gap['name'], gap['name'])
        if gap['name'] == 'illness' and gap['is_covid']:
            name = _gaps['illness_covid']

        if gap['left_edge'] == gap['right_edge']:
            text = string_concat(
                name,
                ' ',
                _('today'),
            )
        else:
            text = string_concat(
                name,
                ' ',
                _('till'),
                ' ',
                gap['right_edge'].day,
                '&nbsp;',
                _(gap['right_edge'].strftime('%B') + 'z'),
            )

        return {'gap': {
            'id': gap['id'],
            'gap_type__color': gap['color'],
            'description': gap['description'],
            'text': text,
            'name': name,
        }}


user_cards = UserCardsView()
