from django.views.decorators.http import require_GET

from staff.person.models.contacts import Contact, ContactTypeId

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.utils.qs_values import extract_related, localize

from staff.person_profile.permissions.utils import observer_has_perm
from staff.person_profile.permissions.properties import Properties
from staff.person_profile.permissions.check_permissions import can_view_phones


def get_contacts(login: str, observer_properties: Properties):
    contacts_qs = (
        Contact.objects.filter(person__login=login)
        .select_related('contact_type')
        .values(
            'contact_type__name',
            'contact_type__name_en',
            'contact_type__icon_url',
            'contact_type__url_pattern',
            'account_id',
            'private',
        )
        .order_by('position')
    )

    if not can_view_phones(None, observer_properties, login):
        contacts_qs = contacts_qs.filter(contact_type_id__in=(
            ContactTypeId.telegram.value, ContactTypeId.assistant.value, ContactTypeId.another_work_email.value
        ))

    result = []

    for contact in contacts_qs:
        contact_type = extract_related(contact, 'contact_type')
        result.append(
            localize({
                'icon_url': contact_type['icon_url'],
                'name': contact_type['name'],
                'name_en': contact_type['name_en'],
                'account_id': contact['account_id'],
                'private': contact['private'],
                'account_url': (
                    contact_type['url_pattern'] % contact['account_id']),
            })
        )
    return result


@available_for_external
@responding_json
@require_GET
@observer_has_perm('contacts')
@use_request_lang
def contacts(request, login):
    result = get_contacts(login=login, observer_properties=request.permissions_ctl.properties)
    if not result:
        return {}
    return {
        'target': {
            'contacts': result
        }
    }
