from django.views.decorators.http import require_http_methods
from django.views.decorators.csrf import ensure_csrf_cookie

from staff.lib.decorators import responding_json, use_request_lang, available_for_external
from staff.lib import waffle

from staff.person_profile.controllers.blocks.meta_blocks_registry import MetaBlocksRegistry
from staff.person_profile.controllers.links_generator import LinksGenerator
from staff.person_profile.controllers.search import find_person, get_search_url
from staff.person_profile.permissions.properties import Properties
from staff.person_profile.permissions.registry import BlockRegistry


@ensure_csrf_cookie
@available_for_external
@responding_json
@require_http_methods(['GET', 'POST'])
@use_request_lang
def meta(request, login):
    vital_switches = {'rkn_mode', 'show_slack_status', 'enable_profile_employment', 'hide_contacts'}
    waffle.preload(BlockRegistry.get_waffle_switches() | vital_switches)

    answer = {}
    observer = request.user.get_profile()
    target = find_person(login, observer, MetaBlocksRegistry.get_fields() | Properties.required_fields | {'tz'})

    if target is None:
        return {'redirect': {'location': get_search_url(login), 'status': 302}}
    elif target['login'] != login:
        login = target['login']
        answer['redirect'] = {'location': '/%s' % login, 'status': 0}

    if login != observer.login and not observer.has_access_to_department_profiles(target['department__id']):
        return {'error': 'You have no access to this profile'}, 403

    properties = Properties(
        target_logins=[login],
        observer=observer,
        readonly=request.service_is_readonly,
        target_data={login: target},
    )

    block_registry = BlockRegistry(properties)

    answer['target'] = MetaBlocksRegistry(target, block_registry).get_data()
    answer['target']['tz'] = target['tz']
    answer['observer'] = {
        'is_owner': block_registry.properties.is_owner(login),
        'hide_middle_name': not observer.show_all_middle_name,
        'tz': observer.tz,
    }

    links = LinksGenerator(target, block_registry).get_links()

    answer['links'] = links
    answer['pencils'] = block_registry.get_pencils(login)
    answer['loadable_blocks'] = block_registry.get_loadable_blocks(login).keys()
    answer['meta_blocks'] = block_registry.get_blocks(login)

    answer['service_is_readonly'] = bool(request.service_is_readonly)

    return answer
