from itertools import groupby
from typing import List, Dict

from django.http import HttpResponse, HttpRequest, HttpResponseNotFound, JsonResponse
from django.views.decorators.http import require_GET
from django.contrib.auth.models import Permission, Group as DjGroup

from staff.groups.models import Group
from staff.person.models import Staff
from staff.users.models import User


def has_perm_directly_assigned(perm: Permission) -> List[str]:
    return list(
        User.objects
        .filter(user_permissions=perm)
        .values_list('username', flat=True),
    )


def has_perm_through_django_groups(perm: Permission) -> Dict[str, List[str]]:
    qs = (
        User.objects
        .filter(groups=DjGroup.objects.filter(permissions=perm))
        .order_by('groups__name', 'username')
        .values_list('groups__name', 'username')
    )
    result = {}
    for group_name, rows in groupby(qs, lambda row: row[0]):
        result[group_name] = [username for _, username in rows]
    return result


def has_perm_through_groups(perm: Permission) -> Dict[str, List[str]]:
    qs = (
        Staff.objects
        .filter(groupmembership__group=Group.objects.filter(permissions=perm))
        .order_by('groupmembership__group__url', 'login')
        .values_list('groupmembership__group__url', 'login')
    )

    result = {}
    for group_name, rows in groupby(qs, lambda row: row[0]):
        result[group_name] = [login for _, login in rows]

    return result


def has_perm_through_department_roles(perm: Permission) -> Dict[str, List[str]]:
    qs = (
        Staff.objects
        .filter(departmentstaff__role__permissions=perm)
        .values_list('departmentstaff__role__name', 'login')
    )

    result = {}
    for role_name, rows in groupby(qs, lambda row: row[0]):
        result[role_name] = [login for _, login in rows]

    return result


@require_GET
def permissions(request: HttpRequest, codename: str) -> HttpResponse:
    perm = Permission.objects.filter(codename=codename).first()
    if not perm:
        return HttpResponseNotFound()

    result = {
        'has_perm_directly_assigned': has_perm_directly_assigned(perm),
        'has_perm_through_django_groups': has_perm_through_django_groups(perm),
        'has_perm_through_groups': has_perm_through_groups(perm),
        'has_perm_through_department_roles': has_perm_through_department_roles(perm)
    }

    return JsonResponse(data=result)
