from collections import defaultdict

from django.views.decorators.http import require_GET

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.utils.qs_values import localize, extract_related

from staff.person_profile.permissions.utils import observer_has_perm

from staff.person.models import ResponsibleForRobot


def get_responsible_for_robot(login):
    qs = (
        ResponsibleForRobot.objects
        .filter(robot__login=login)
        .values(
            'responsible__login',
            'responsible__first_name',
            'responsible__last_name',
            'responsible__first_name_en',
            'responsible__last_name_en',
            'role',
        )
    )

    result = defaultdict(list)
    for role in qs:
        role = localize(role)
        responsible = extract_related(role, 'responsible')
        result[role['role']].append(responsible)

    return result


@require_GET
@available_for_external
@responding_json
@observer_has_perm('responsible_for_robot')
@use_request_lang
def responsible_for_robot(request, login):
    result = get_responsible_for_robot(login)
    if result:
        return {'target': {'responsible_for_robot': result}}
    return {}
