import json
import logging
import requests
import waffle

from django.conf import settings
from django.http import JsonResponse
from django.views.decorators.http import require_http_methods

from staff.lib.decorators import available_for_external

logger = logging.getLogger(__name__)


@available_for_external
@require_http_methods(['GET', 'POST'])
def status(request, login):
    url = f'https://{settings.SLACK_HOST}/ya-admin/get-status/'

    if not waffle.switch_is_active('show_slack_status'):
        return JsonResponse({'status': settings.SLACK_STATUS_DISABLED}, status=200)

    try:
        response = requests.get(url, params={'username': login}, timeout=0.15)
        response.raise_for_status()
    except requests.Timeout:
        logger.info(
            'Can not get slack activity, %s timed out',
            settings.SLACK_HOST,
            exc_info=True,
        )
        return JsonResponse({'error': f'{settings.SLACK_HOST} does not responding'}, status=200)

    except requests.RequestException:
        logger.info(
            'Can not get slack activity: %s responded with status %s',
            settings.SLACK_HOST,
            response.status_code,
            exc_info=True,
        )
        return JsonResponse(
            {
                'error': f'{settings.SLACK_HOST} responded with status {response.status_code}',
            },
            status=200,
        )

    try:
        return JsonResponse({'target': {'slack_status': response.json()}}, status=200)
    except (TypeError, json.JSONDecodeError):
        return JsonResponse({'error': f'{settings.SLACK_HOST} sent invalid json'}, status=200)
