from datetime import datetime

from django.views.decorators.http import require_GET

from django.db.models import Q

from staff.departments.utils import get_department_chain
from staff.lib.utils.qs_values import extract_related, localize
from staff.person.models import Staff
from staff.survey.models import Survey

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from ..permissions.utils import observer_has_perm


def get_actual_surveys(login):
    now = datetime.now()

    staff = (
        Staff.objects
        .values(
            'login',
            'office_id',
            'department_id',
            'department__lft',
            'department__rght',
            'department__tree_id',
        )
        .get(login=login)
    )

    department = extract_related(staff, 'department')
    dep_ids = [dep['id'] for dep in get_department_chain(**department)]

    return list(
        Survey.objects
        .filter(Q(department_id__in=dep_ids) | Q(department__isnull=True))
        .filter(Q(office_id=staff['office_id']) | Q(office__isnull=True))
        .filter(start_at__lte=now, end_at__gte=now)
        .order_by('-created_at')
        .values(
            'id',
            'block_text',
            'block_text_en',
            'button_text',
            'button_text_en',
            'button_link',
            'button_link_en'
        )
    )


@require_GET
@available_for_external
@responding_json
@observer_has_perm('survey')
@use_request_lang
def survey(request, login):
    result = [localize(survey) for survey in get_actual_surveys(login)]
    if not result:
        return {}
    return {'target': {'surveys': result}}
