import logging
from datetime import date

from django.views.decorators.http import require_GET

from staff.lib.utils.qs_values import localize, extract_related
from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.forms.errors import default_error
from staff.person_profile.permissions.utils import observer_has_perm
from staff.map.models import TableBook

logger = logging.getLogger('person_profile.table_book_view')


@available_for_external
@responding_json
@require_GET
@observer_has_perm('table_books')
@use_request_lang
def table_books(request, login):
    try:
        table_books = (
            TableBook.objects
            .filter(staff__login=login, date_to__gte=date.today)
            .order_by('date_from')
            .values(
                'table_id',
                'description',
                'table__floor__name',
                'table__floor__name_en',
                'table__floor_id',
                'table__floor__office__code',
                'table__floor__office__name',
                'table__floor__office__name_en',
                'date_from',
                'date_to',
                'table__room__name',
                'table__room__name_en',
                'table__room_id',
            )
        )
    except Exception:
        logger.exception('Error while retrieving TableBook data')
        return default_error(), 500

    def normalize(table_book):
        localized = localize(table_book)
        localized.update(
            extract_related(localized, 'table__floor__office', prefix='office')
        )
        localized.update(
            extract_related(localized, 'table__floor', prefix='floor')
        )
        localized.update(
            extract_related(localized, 'table__room', prefix='room')
        )
        return localized

    if table_books:
        return {'target': {'table_books':
                [normalize(table_book) for table_book in table_books]}}
    return {}
