from django.contrib import admin
from django import forms

from staff.lib.utils.admin import BaseIntranetAdmin, AutocompleteMixin

from staff.preprofile.models import (
    Preprofile,
    PreprofileABCServices,
    PersonAdoptApplication,
    IllegalLogin,
    HardwareProfile,
    ProfileForDepartment,
    NamePreposition,
)


class IllegalLoginAdmin(admin.ModelAdmin):
    list_editable = ('login', 'exact_match')
    list_display = list_editable
    search_fields = list_editable


class NamePrepositionAdmin(admin.ModelAdmin):
    list_editable = 'preposition',
    list_display = list_editable
    search_fields = list_editable


class PersonAdoptApplicationAdmin(admin.ModelAdmin, AutocompleteMixin):
    staff_field = 'adopter'
    list_display = (
        'login',
        'first_name',
        'last_name',
        'offer_id',
        'status',
        'status_reason',
    )
    search_fields = (
        'login',
        'first_name',
        'last_name',
    )
    list_filter = ('status', 'status_reason',)
    raw_id_fields = 'preprofile',

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        autocomplete_field = self.maybe_autocomplete_formfield(db_field)
        if autocomplete_field is not None:
            return autocomplete_field

        return super(PersonAdoptApplicationAdmin, self).formfield_for_foreignkey(
            db_field, request, **kwargs)


class PreprofileABCServicesInline(admin.TabularInline, AutocompleteMixin):
    model = PreprofileABCServices
    extra = 0
    group_field = 'group'

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        autocomplete_field = self.maybe_autocomplete_formfield(db_field)
        if autocomplete_field is not None:
            return autocomplete_field

        return super(PreprofileABCServicesInline, self).formfield_for_foreignkey(
            db_field, request, **kwargs)


class PreprofileModelForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super(PreprofileModelForm, self).__init__(*args, **kwargs)
        for field in self.fields.values():
            field.required = False

    class Meta:
        fields = '__all__'
        models = Preprofile


class PreprofileAdmin(BaseIntranetAdmin, AutocompleteMixin):
    form = PreprofileModelForm

    department_field = ('department',)
    staff_field = (
        'mentor',
        'robot_owner',
        'recruiter',
        'approved_by',
        'adopted_by',
    )
    raw_id_fields = ('table', 'room')

    inlines = (PreprofileABCServicesInline,)
    list_display = (
        'login',
        'first_name',
        'last_name',
        'join_at',
        'form_type',
        'status',
        'candidate_type',
    )
    search_fields = (
        'login',
        'first_name',
        'first_name_en',
        'last_name',
        'last_name_en',
        'femida_offer_id',
        'masshire_tag',
    )
    list_filter = ('form_type', 'status', 'candidate_type')
    readonly_fields = (
        'created_at',
        'modified_at',
    )
    fieldsets = (
        ('State', {'fields': (
            ('status', 'form_type', 'candidate_type'),
            ('approved_by', 'adopted_by'),
            'ext_form_link',
            ('created_at', 'modified_at'),
            'join_at',
        )}),
        ('ID', {'fields': (
            ('login', 'uid', 'guid'),
            ('crm_alias', ),
            ('masshire_tag', ),
        )}),
        ('Name', {'classes': ('collapse',), 'fields': (
            ('last_name', 'last_name_en'),
            ('first_name', 'first_name_en'),
            'middle_name',
        )}),
        ('Hardware', {'classes': ('collapse', ), 'fields': (
            'need_vpn',
            'os',
            'docs_and_hw_address',
            'hardware_office',
            'hardware_profile',
            'hardware_profile_description',
            'hardware_profile_needs_table',
        )}),
        ('Location', {'classes': ('collapse', ), 'fields': (
            'office',
            'table',
            'room',
            'need_table',
            'need_chair',
            'need_phone',
        )}),
        ('Femida', {'classes': ('collapse',), 'fields': (
            'femida_offer_id',
            'recruiter',
        )}),
        ('Tickets', {'classes': ('collapse',), 'fields': (
            'hr_ticket',
            'adopt_ticket',
            'hdrfs_ticket',
            'supply_ticket',
            'tools_ticket',
        )}),
        ('Robot', {'classes': ('collapse',), 'fields': (
            'robot_owner',
            'zombie_hw',
            'remote_access',
            'network_connected',
            'need_tv',
        )}),
        ('Official', {'classes': ('collapse', ), 'fields': (
            'department',
            'organization',
            ('position', 'position_staff_text'),
            ('duties', 'duties_en'),
            'employment_type',
            'email_domain',
            'mentor',
            'lang_ui',
            'citizenship',
            'date_completion_internship',
        )}),
        ('Personal', {'classes': ('collapse', ), 'fields': (
            'photo',
            'birthday',
            'gender',
            'phone',
            'email',
            'address',
        )}),
    )

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        autocomplete_field = self.maybe_autocomplete_formfield(db_field)
        if autocomplete_field is not None:
            return autocomplete_field

        return super(PreprofileAdmin, self).formfield_for_foreignkey(
            db_field, request, **kwargs)


class HardwareProfileAdmin(admin.ModelAdmin):
    list_display = (
        'profile_id',
        'name',
    )
    search_fields = list_display

    def get_search_results(self, request, queryset, search_term):
        queryset, use_distinct = super(HardwareProfileAdmin, self).get_search_results(request, queryset, search_term)

        profile_ids = (
            ProfileForDepartment.objects
            .filter(department__url__in=search_term.split(' '))
            .values_list('profile__profile_id', flat=True)
        )

        extra_queryset = HardwareProfile.objects.filter(profile_id__in=profile_ids)
        new_queryset = queryset | extra_queryset

        return new_queryset, use_distinct


admin.site.register(Preprofile, PreprofileAdmin)
admin.site.register(PersonAdoptApplication, PersonAdoptApplicationAdmin)
admin.site.register(IllegalLogin, IllegalLoginAdmin)
admin.site.register(NamePreposition, NamePrepositionAdmin)
admin.site.register(HardwareProfile, HardwareProfileAdmin)
