from django.conf import settings

from staff.preprofile.utils import launch_person_avatar_task
from staff.person.models import EMPLOYMENT
from staff.preprofile.controller_behaviours import AbstractBehaviour
from staff.preprofile.forms import OutstaffForm
from staff.preprofile.models import PREPROFILE_STATUS, FORM_TYPE, EMAIL_DOMAIN


class OutstaffBehaviour(AbstractBehaviour):
    def create_form(self, action_context, data=None):
        return OutstaffForm(data=data, base_initial={'action_context': action_context})

    def create_edit_form(self, action_context, initial_data, data=None):
        return OutstaffForm(initial=initial_data, data=data, base_initial={'action_context': action_context})

    def actions_on_create(self):
        return {'save': True, 'approve': False, 'adopt': False, 'create_link': False}

    def actions_on_edit(self, action_context):
        in_approvable_status = action_context.preprofile.status in [
            PREPROFILE_STATUS.NEW,
            PREPROFILE_STATUS.PREPARED,
        ]
        preprofile_has_valid_login = bool(action_context.preprofile.login and action_context.preprofile.candidate_type)
        preprofile_approvable = in_approvable_status and preprofile_has_valid_login

        adoptable = action_context.preprofile.status == PREPROFILE_STATUS.READY
        can_create_link = in_approvable_status
        return {'save': True, 'approve': preprofile_approvable, 'adopt': adoptable, 'create_link': can_create_link}

    def form_type(self):
        return FORM_TYPE.OUTSTAFF

    def on_new_model(self, action_context):
        action_context.preprofile.email_domain = EMAIL_DOMAIN.YANDEX_TEAM_RU
        action_context.preprofile.status = PREPROFILE_STATUS.NEW
        action_context.preprofile.employment_type = EMPLOYMENT.FULL
        action_context.preprofile.office_id = settings.HOMIE_OFFICE_ID

    def is_editable(self):
        return True

    def is_cancelable(self):
        return True

    def is_prepareable(self):
        return True

    def is_approvable(self, action_context):
        return action_context.preprofile.login and action_context.preprofile.candidate_type

    def on_after_create(self, action_context):
        launch_person_avatar_task(task_kwargs={'preprofile_ids': [action_context.preprofile.id]})

    def on_after_change(self, action_context):
        pass


AbstractBehaviour.register(OutstaffBehaviour)
