from staff.person.models import Staff
from staff.preprofile.action_context import ActionContext
from staff.preprofile.controller_behaviours import AbstractBehaviour
from staff.preprofile.forms import (
    RotationForm,
    EmployeeEditForm,
)
from staff.preprofile.models import (
    CANDIDATE_TYPE,
    FORM_TYPE,
    PREPROFILE_STATUS,
)


class RotationBehaviour(AbstractBehaviour):

    def create_form(self, action_context, data=None):
        return RotationForm(data=data)

    def create_edit_form(self, action_context, initial_data, data=None):
        return EmployeeEditForm(initial=initial_data, data=data, base_initial={'action_context': action_context})

    def actions_on_create(self):
        pass

    def actions_on_edit(self, action_context):
        preprofile = action_context.preprofile

        is_new = preprofile.status == PREPROFILE_STATUS.NEW
        can_save = (
            is_new or
            preprofile.status in (
                PREPROFILE_STATUS.NEW,
                PREPROFILE_STATUS.APPROVED,
                PREPROFILE_STATUS.READY,
            ) and
            action_context.preprofile.recruiter == action_context.requested_by
        )
        return {
            'save': can_save,
            'approve': is_new,
            'adopt': False,
            'create_link': False,
        }

    def form_type(self):
        return FORM_TYPE.ROTATION

    def on_new_model(self, action_context):
        action_context.preprofile.candidate_type = CANDIDATE_TYPE.CURRENT_EMPLOYEE

    def is_editable(self):
        return True

    def is_cancelable(self):
        return False

    def is_prepareable(self):
        return False

    def is_approvable(self, action_context):
        return True

    def on_after_create(self, action_context):
        # type: (ActionContext) -> None
        preprofile = action_context.preprofile
        person = Staff.objects.get(login=preprofile.login)

        preprofile.uid = person.uid
        preprofile.guid = person.guid
        preprofile.first_name = person.first_name
        preprofile.first_name_en = person.first_name_en
        preprofile.middle_name = person.middle_name
        preprofile.last_name = person.last_name
        preprofile.last_name_en = person.last_name_en
        preprofile.birthday = person.birthday
        preprofile.gender = person.gender
        preprofile.address = person.address

        preprofile.phone = person.mobile_phone
        preprofile.email = person.home_email
        if not preprofile.organization:
            preprofile.organization = person.organization
        if not preprofile.office:
            preprofile.office = person.office

        preprofile.save()

    def on_after_change(self, action_context):
        from staff.preprofile.tasks import AutoAdoptPreprofiles
        AutoAdoptPreprofiles(preprofile_id=action_context.preprofile.id)


AbstractBehaviour.register(RotationBehaviour)
