import logging
from datetime import date

from staff.achievery.internship import give_internship_achievement
from staff.departments.models import DepartmentRoles
from staff.lib.db import atomic
from staff.lib.models.roles_chain import has_roles_for_department
from staff.person.models import Staff

from staff.preprofile.action_context import ActionContext
from staff.preprofile.adopt_api import adopt_preprofile
from staff.preprofile.controllers.errors import AccessDeniedControllerError, raise_controller_error
from staff.preprofile.models import CANDIDATE_TYPE, FORM_TYPE, PREPROFILE_STATUS
from staff.preprofile.notifications import notify_preprofile_adopted
from staff.preprofile.utils import user_is_adopter, outstaff_or_external_department


logger = logging.getLogger(__name__)


class AdoptController(object):
    def __init__(self, action_context: ActionContext):
        assert action_context.preprofile and action_context.requested_by
        self._preprofile = action_context.preprofile
        self._requested_by = action_context.requested_by

    @atomic
    def adopt(self):
        if self._moving_fired_employee():
            raise_controller_error('moving_fired_employee')

        if not self.user_has_right_to_adopt():
            logging.info('User %s has no rights for adopt %s', self._requested_by.login, self._preprofile.login)
            raise AccessDeniedControllerError()

        if not self._status_is_ready():
            raise_controller_error('not_applicable')

        if self.join_date_in_future():
            raise_controller_error('join_date_in_future')

        if not self._has_active_department():
            raise_controller_error('inactive_department')

        logging.info(
            'Adopting preprofile for %s by %s',
            self._preprofile.login,
            self._requested_by.login,
        )

        adopt_preprofile(self._preprofile, self._requested_by)

        self._preprofile.adopted_by = self._requested_by
        self._preprofile.status = PREPROFILE_STATUS.CLOSED
        self._preprofile.save()
        self._run_adopt_side_effects()

    def join_date_in_future(self):
        return date.today() < self._preprofile.join_at

    def user_has_right_to_adopt(self):
        if user_is_adopter(self._requested_by):
            return True

        department = self._preprofile.department

        if self._person_is_chief_and_has_special_adoption_right(department):
            return True

        if (
            self._preprofile.form_type == FORM_TYPE.EMPLOYEE
            and outstaff_or_external_department(department)
            and self._person_is_chief_or_deputy(department)
        ):
            return True

        if self._preprofile.form_type in (FORM_TYPE.OUTSTAFF, FORM_TYPE.EXTERNAL):
            return self._person_is_chief_or_deputy(department)

        return False

    def user_can_adopt(self):
        return (
            not self._moving_fired_employee()
            and self.user_has_right_to_adopt()
            and self._status_is_ready()
            and not self.join_date_in_future()
        )

    def _status_is_ready(self):
        return self._preprofile.status == PREPROFILE_STATUS.READY

    def _has_active_department(self):
        return self._preprofile.department.intranet_status == 1

    def _moving_fired_employee(self):
        if self._preprofile.candidate_type != CANDIDATE_TYPE.EXTERNAL_EMPLOYEE:
            return False

        person = Staff.objects.get(login=self._preprofile.login)
        return person.is_dismissed

    def _person_is_chief(self, department):
        is_chief = has_roles_for_department(department, self._requested_by, [DepartmentRoles.CHIEF.value])
        return is_chief

    def _person_is_chief_or_deputy(self, department):
        roles = [DepartmentRoles.CHIEF.value, DepartmentRoles.DEPUTY.value]
        return has_roles_for_department(department, self._requested_by, roles)

    def _person_is_chief_and_has_special_adoption_right(self, department):
        return self._person_is_chief(department) and self._requested_by.user.has_perm('preprofile.chief_that_can_adopt')

    def _run_adopt_side_effects(self):
        notify_preprofile_adopted(self._preprofile)
        if (
            self._preprofile.date_completion_internship is not None
            and self._preprofile.date_completion_internship > date.today()
        ):
            give_internship_achievement(self._preprofile)
