import datetime
import logging
from typing import Dict, Any, Optional, List

import yenv

from django.conf import settings

from staff.person.models import Staff
from staff.person_avatar.tasks import change_preprofile_login
from staff.lib import requests
from staff.lib.models.roles_chain import get_hrbp_by_departments, direct_chief_for_department

from staff.preprofile import notifications
from staff.preprofile.action_context import ActionContext
from staff.preprofile.controllers.adopt_controller import AdoptController
from staff.preprofile.controllers.errors import ControllerError, raise_controller_error
from staff.preprofile.controllers.name_fixer import NameFixer
from staff.preprofile.controller_behaviours import AbstractBehaviour
from staff.preprofile.ldap_utils import get_guid_from_ldap
from staff.preprofile.models import (
    Preprofile,
    PREPROFILE_STATUS,
    PreprofileABCServices,
    CANDIDATE_TYPE,
)
from staff.preprofile.utils import get_uid_from_passport

logger = logging.getLogger(__name__)


_MANUAL_COPY_FIELDS = {
    'status',
    'abc_services',
    'department',
    'organization',
    'office',
    'hardware_office',
    'table',
    'room',
    'recruiter',
    'approved_by',
    'adopted_by',
}


_FIELDS_FOR_COPY = [
    f.name
    for f in Preprofile._meta.fields
    if f.name not in _MANUAL_COPY_FIELDS
]


_MANUAL_COPY_FIELDS_FROM_FORM_DATA = {
    'abc_services',
    'approved_by',
}


_FIELDS_FOR_COPY_FROM_FORM_DATA = [
    f.name
    for f in Preprofile._meta.fields
    if f.name not in _MANUAL_COPY_FIELDS_FROM_FORM_DATA
]


class Controller(object):
    _name_fields = ['first_name', 'first_name_en', 'middle_name', 'last_name', 'last_name_en']

    def __init__(
        self,
        behaviour: AbstractBehaviour,
        action_context: ActionContext,
        name_fixer: Optional[NameFixer] = None,
    ):
        self._action_context = action_context
        self._behaviour = behaviour
        self._name_fixer = name_fixer or NameFixer()

    def try_apply_changes(self, form_data: Dict[str, Any]):
        if self.is_editing and self.status in (PREPROFILE_STATUS.CLOSED, PREPROFILE_STATUS.CANCELLED):
            raise_controller_error('not_editable')

        if self.is_editing and not self._behaviour.is_editable():
            raise_controller_error('cant_edit_robot')

        form = self._create_form(data=form_data)

        if not form.is_valid():
            raise ControllerError(errors_dict=form.errors_as_dict())

        was_created = False
        if not self.is_editing:
            self._action_context.preprofile = self._new_preprofile_model()
            self._behaviour.on_new_model(self._action_context)
            was_created = True

        form_data = form.cleaned_data

        changed = self._copy_fields_from_form_to_model(form_data)
        self._fix_names()
        self._save()

        self._copy_abc_services_from_form_to_model(form_data)
        self._save()

        old_login = changed.get('login', {}).get('old')
        self._on_login_changed_on_edit(old_login)

        if was_created:
            self._behaviour.on_after_create(self._action_context)
            if self.is_autohire:
                self.approve()
        else:
            self._behaviour.on_after_change(self._action_context)

        self._send_notifications(was_created, changed)

    def _fix_names(self):
        pre_profile = self._action_context.preprofile

        for name_field in self._name_fields:
            current_name = getattr(pre_profile, name_field)
            fixed_name = self._name_fixer.fix_name(current_name)

            if current_name != fixed_name:
                setattr(pre_profile, name_field, fixed_name)
                logger.info(
                    '%s name %s fixed from "%s" to "%s"',
                    pre_profile.login,
                    name_field,
                    current_name,
                    fixed_name,
                )

    def _send_notifications(self, was_created, changed):
        if not was_created:
            notifications.notify_preprofile_changed(self._action_context.preprofile, changed)
            return

        notifications.notify_preprofile_created(self._action_context.preprofile)
        if self.status == PREPROFILE_STATUS.APPROVED and not self.is_autohire:
            notifications.notify_preprofile_approved(self._action_context.preprofile)

    def _on_login_changed_on_edit(self, old_login):
        if old_login != self._action_context.preprofile.login:
            return

        task_kwargs = {
            'preprofile_id': self.preprofile_id,
            'old_login': old_login,
        }

        task = change_preprofile_login
        if yenv.type == 'development':
            task(**task_kwargs)
        else:
            task.apply_async(kwargs=task_kwargs, countdown=20)

    @property
    def status(self):
        return self._action_context.preprofile.status

    @property
    def form_type(self):
        return self._action_context.preprofile.form_type

    @property
    def preprofile_id(self):
        return self._action_context.preprofile.id

    @property
    def is_autohire(self):
        return self._action_context.preprofile.is_autohire

    @property
    def department(self):
        return self._action_context.preprofile.department

    @property
    def is_editing(self):
        """
        :rtype: bool
        """
        return self._action_context.preprofile is not None

    @property
    def requested_by(self):
        return self._action_context.requested_by

    @property
    def preprofile_was_created_by_femida(self):
        """
        :rtype: bool
        """
        return self.is_editing and self._action_context.preprofile.femida_offer_id

    def _create_form(self, data=None):
        if self.is_editing:
            return self._behaviour.create_edit_form(
                action_context=self._action_context,
                initial_data=self._initial_data_from_model(),
                data=data,
            )

        return self._behaviour.create_form(
            action_context=self._action_context,
            data=data,
        )

    def front_data(self):
        form_data = self._create_form().as_dict()
        form_data['form_type'] = self._behaviour.form_type()

        if self.is_editing:
            form_data['actions'] = self._behaviour.actions_on_edit(self._action_context)
            return form_data

        form_data['actions'] = self._behaviour.actions_on_create()
        return form_data

    def _new_preprofile_model(self) -> Preprofile:
        form_type = self._behaviour.form_type()
        result = Preprofile(recruiter=self._action_context.requested_by)
        result.status = PREPROFILE_STATUS.NEW
        result.form_type = form_type

        return result

    def _try_copy_field(self, field, nested_field, dst):
        """
        :type field: str
        :type field: nested_field
        :type dst: dict
        """
        if not hasattr(self._action_context.preprofile, field):
            return

        field_value = self._action_context.preprofile.__getattribute__(field)
        if not field_value:
            return

        dst[field] = field_value.__getattribute__(nested_field)

    def _initial_data_from_model(self):
        """
        :rtype: dict
        """

        data = {}

        for field in _FIELDS_FOR_COPY:
            if hasattr(self._action_context.preprofile, field):
                data[field] = self._action_context.preprofile.__getattribute__(field)

        data['id'] = self.preprofile_id
        data['status'] = self.status
        data['abc_services'] = self.abc_services()
        data['department'] = self.department.url
        data['chief'] = self.chief_login()
        data['hr_partners'] = self.hr_partners()

        self._try_copy_field('organization', 'id', data)
        self._try_copy_field('office', 'id', data)
        self._try_copy_field('hardware_office', 'id', data)
        self._try_copy_field('table', 'id', data)
        self._try_copy_field('room', 'id', data)
        self._try_copy_field('recruiter', 'login', data)
        self._try_copy_field('approved_by', 'login', data)
        self._try_copy_field('adopted_by', 'login', data)

        if self._action_context.preprofile.robot_owner:
            data['responsible'] = self._action_context.preprofile.robot_owner.login

        return data

    def chief_login(self):
        """
        :rtype: str
        """
        chief = direct_chief_for_department(self.department, fields=['login'])
        return chief['login'] if chief else ''

    def hr_partners(self):
        """
        :rtype: list
        """
        return [
            partner['login']
            for partner in get_hrbp_by_departments([self.department], fields=['login'])
        ]

    def abc_services(self) -> List[str]:
        return [service.code for service in self._action_context.preprofile.abc_services.all()]

    def _copy_fields_from_form_to_model(self, form_data):
        """
        :type form_data: dict
        """
        changed = {}
        preprofile = self._action_context.preprofile
        for field in _FIELDS_FOR_COPY_FROM_FORM_DATA:
            if field in form_data:
                new = form_data[field]
                changed[field] = {
                    'old': getattr(preprofile, field, None),
                    'new': new,
                }
                preprofile.__setattr__(field, new)

        if 'responsible' in form_data:
            new = form_data['responsible']
            changed['responsible'] = {
                'old': preprofile.robot_owner,
                'new': new,
            }
            preprofile.robot_owner = new
        return changed

    def _copy_abc_services_from_form_to_model(self, form_data):
        """
        :type form_data: dict
        ManyToMany fields requires already saved model
        """
        self._requires_saved_model()

        if 'abc_services' in form_data:
            PreprofileABCServices.objects.filter(preprofile=self._action_context.preprofile).delete()
            PreprofileABCServices.objects.bulk_create(
                PreprofileABCServices(group=g, preprofile=self._action_context.preprofile)
                for g in form_data['abc_services']
            )

    def _create_external_link(self):
        femida_create_link_url = 'https://{host}{url}'.format(
            host=settings.FEMIDA_HOST,
            url='/_api/newhire/preprofiles/',
        )
        try:
            response_dict = requests.post(
                femida_create_link_url,
                headers={'Authorization': 'OAuth {}'.format(settings.ROBOT_STAFF_OAUTH_TOKEN)},
                json={'id': self.preprofile_id},
                timeout=(0.5, 1, 3),
            ).json()
            return response_dict['url']
        except requests.RequestException:
            logger.exception(
                'Error trying to create external link for %s.',
                self.preprofile_id,
            )
            raise_controller_error('femida_error')

    def create_link(self):
        if self.status not in [PREPROFILE_STATUS.NEW, PREPROFILE_STATUS.PREPARED]:
            raise_controller_error('not_applicable')

        self._action_context.preprofile.ext_form_link = self._create_external_link()
        self._action_context.preprofile.save()
        return self._action_context.preprofile.ext_form_link

    def prepare(self):
        self._requires_saved_model()

        if not self._behaviour.is_prepareable():
            raise_controller_error('cant_be_prepared')

        self._save(PREPROFILE_STATUS.PREPARED)
        notifications.notify_preprofile_prepared(self._action_context.preprofile)

    def cancel(self):
        self._requires_saved_model()

        if not self._behaviour.is_cancelable():
            raise_controller_error('cant_be_canceled')

        self._save(PREPROFILE_STATUS.CANCELLED)
        notifications.notify_preprofile_cancelled(self._action_context.preprofile)

    def approve(self, join_at=None):
        self._requires_saved_model()

        if self.status not in [PREPROFILE_STATUS.NEW, PREPROFILE_STATUS.PREPARED]:
            raise_controller_error('not_applicable')

        if not self._behaviour.is_approvable(self._action_context):
            raise_controller_error('not_applicable')

        logging.info(
            'Approving preprofile for %s by %s',
            self._action_context.preprofile.login,
            self._action_context.requested_by.login,
        )

        if join_at:
            self._action_context.preprofile.join_at = join_at

        self._action_context.preprofile.approved_by = self._action_context.requested_by
        self._save(PREPROFILE_STATUS.APPROVED)

        notifications.notify_preprofile_approved(self._action_context.preprofile)

        if self._preprofile_with_existing_login():
            person = Staff.objects.get(login=self._action_context.preprofile.login)
            self._action_context.preprofile.uid = person.uid
            self._action_context.preprofile.guid = person.guid
            self._save(PREPROFILE_STATUS.READY)

    def adopt(self):
        self._requires_saved_model()
        adopt_controller = AdoptController(self._action_context)
        adopt_controller.adopt()

    def can_be_adopted(self):
        self._requires_saved_model()
        adopt_controller = AdoptController(self._action_context)
        return adopt_controller.user_can_adopt()

    def try_make_ready(self) -> bool:
        self._requires_saved_model()

        if self.status != PREPROFILE_STATUS.APPROVED:
            raise_controller_error('not_applicable')

        preprofile = self._action_context.preprofile

        if not preprofile.uid:
            preprofile.uid = get_uid_from_passport(preprofile.login)

            if not preprofile.uid and preprofile.join_at < datetime.date.today():
                logger.info(f'Preprofile {preprofile.id} ({preprofile.login}) passport uid still empty')

        if not preprofile.guid:
            preprofile.guid = get_guid_from_ldap(preprofile.login)

            if not preprofile.guid and preprofile.join_at < datetime.date.today():
                logger.info(f'Preprofile {preprofile.id} ({preprofile.login}) ldap guid still empty')

        if preprofile.uid and preprofile.guid:
            self._save(PREPROFILE_STATUS.READY)
            return True

        return False

    def _requires_saved_model(self):
        if not self.preprofile_id:
            raise_controller_error('there_is_no_preprofile')

    def _save(self, new_status=None):
        if new_status:
            self._action_context.preprofile.status = new_status

        self._action_context.preprofile.save()

    def _preprofile_with_existing_login(self):
        return self._action_context.preprofile.candidate_type in [
            CANDIDATE_TYPE.FORMER_EMPLOYEE,
            CANDIDATE_TYPE.EXTERNAL_EMPLOYEE,
            CANDIDATE_TYPE.CURRENT_EMPLOYEE,
        ]
