import re
from typing import Dict, Optional, Pattern

from staff.preprofile.models import NamePreposition


class NameFixer:
    _exclusion_mapping_cache: Optional[Dict[str, str]] = None
    _name_pattern: Pattern = re.compile('([\\w\u0301]+)')
    _solid_name_pattern: Pattern = re.compile('[^\\w\u0301]')

    def fix_name(self, value: str) -> str:
        if not value:
            return value

        if not self._solid_name_pattern.findall(value):
            return value[0].upper() + value[1:].lower()

        def _convert_to_correct(match_obj):
            if match_obj.group() is not None:
                caught = match_obj.group()
                return self._get_exclusions().get(caught.lower(), caught[0].upper() + caught[1:].lower())

        return self._name_pattern.sub(_convert_to_correct, value)

    def _get_exclusions(self) -> Dict[str, str]:
        if not self._exclusion_mapping_cache:
            exclusions = NamePreposition.objects.all().values_list('preposition', flat=True)
            self._exclusion_mapping_cache = {value.lower(): value for value in exclusions}

        return self._exclusion_mapping_cache
