import sform

from staff.person_avatar.controllers.preprofile import PreprofileAvatarCollection

from staff.preprofile.action_context import ActionContext
from staff.preprofile.forms.fields import (
    contacts_fields,
    helpdesk_fields,
    official_fields,
    personal_fields,
    technical_fields,
)
from staff.preprofile.forms.utils import remove_readonly_fields
from staff.preprofile.forms.validation import (
    validate_hardware_profile_by_department,
    validate_login_with_candidate_type,
    validate_room_with_candidate_type,
)
from staff.preprofile.models import (
    FORM_TYPE,
    PREPROFILE_STATUS,
)
from staff.preprofile.utils import user_is_adopter


ENABLED_FIELDS = {
    'login',
    'join_at',
    'need_table',
    'need_chair',
    'need_vpn',
    'crm_alias',
    'lang_ui',
    'table',
    'room',
    'hardware_profile',
    'hardware_profile_description',
    'abc_services',
    'need_table',
    'need_table',
    'need_chair',
    'need_phone',
}

ENABLED_FIELDS_FOR_RECRUITER = {
    'first_name_en',
    'first_name',
    'last_name_en',
    'last_name',
    'middle_name',
    'gender',
    'birthday',
    'phone',
    'email',
    'address',
    'organization',
    'office',
    'position_staff_text',
    'os',
}


class EmployeeEditForm(sform.SForm):
    first_name_en = personal_fields.first_name_en()
    last_name_en = personal_fields.last_name_en()
    first_name = personal_fields.first_name()
    last_name = personal_fields.last_name()
    gender = personal_fields.gender()
    birthday = personal_fields.birthday(required=False)
    middle_name = personal_fields.middle_name(required=False)
    photo = personal_fields.photo(required=False)

    position = official_fields.position(required=False)
    department = official_fields.department()
    office = official_fields.office()
    join_at = official_fields.join_at()
    organization = official_fields.organization()
    candidate_type = official_fields.candidate_type()
    employment_type = official_fields.employment_type()
    abc_services = official_fields.abc_services(required=False)
    position_staff_text = official_fields.position_staff_text()
    login = official_fields.human_login()

    phone = contacts_fields.phone(required=False)
    email = contacts_fields.email(required=False)
    address = contacts_fields.address(required=False)

    hardware_profile = helpdesk_fields.hardware_profile()
    hardware_profile_description = helpdesk_fields.hardware_profile_description(required=False)

    need_table = helpdesk_fields.need_table()
    need_chair = helpdesk_fields.need_chair()
    need_phone = helpdesk_fields.need_phone()
    need_vpn = helpdesk_fields.need_vpn(required=False)
    crm_alias = helpdesk_fields.crm_alias(required=False)
    lang_ui = helpdesk_fields.lang_ui(required=False)
    table = helpdesk_fields.table(required=False)
    room = helpdesk_fields.room(required=False)
    os = helpdesk_fields.os(readonly=True)

    uid = technical_fields.uid(readonly=True)
    guid = technical_fields.guid(readonly=True)
    created_at = technical_fields.created_at(readonly=True)
    modified_at = technical_fields.modified_at(readonly=True)
    status = technical_fields.status(readonly=True)
    chief = technical_fields.chief(readonly=True)
    hr_partners = technical_fields.hr_partners(readonly=True)
    recruiter = technical_fields.recruiter(readonly=True)
    adopted_by = technical_fields.adopted_by(readonly=True)
    approved_by = technical_fields.approved_by(readonly=True)

    @property
    def action_context(self) -> ActionContext:
        return self.base_initial['action_context']

    def in_editable_state(self):
        readonly = [PREPROFILE_STATUS.CLOSED, PREPROFILE_STATUS.CANCELLED]
        return self.action_context.preprofile.status not in readonly

    def in_not_editable_state_for_non_recruiter(self):
        not_editable_states_for_non_recruiter = [PREPROFILE_STATUS.APPROVED, PREPROFILE_STATUS.READY]
        return self.action_context.preprofile.status in not_editable_states_for_non_recruiter

    def editing_login_when_has_uid_or_guid(self, name):
        return name == 'login' and self.action_context.preprofile.has_uid_or_guid

    def editing_by_recruiter(self):
        return self.action_context.preprofile.recruiter == self.action_context.requested_by

    def editing_join_at_by_adopter(self, name):
        return name == 'join_at' and user_is_adopter(self.action_context.requested_by)

    def editing_verified_phone(self, name):
        return name == 'phone' and self.action_context.preprofile.is_eds_phone_verified

    def is_rotation(self):
        return self.action_context.preprofile.form_type == FORM_TYPE.ROTATION

    # STAFF-15775
    # def is_intern(self):
    #     return self.action_context.preprofile.date_completion_internship is not None

    def get_field_state(self, name):
        if self.is_rotation() and name == 'join_at':
            is_editable = (
                    self.editing_by_recruiter()
                    and self.in_editable_state()
            )
            return sform.REQUIRED if is_editable else sform.READONLY

        # STAFF-15775
        # if (name in ['need_table', 'need_chair', 'table', 'room']) and self.is_intern():
        #     return sform.READONLY

        if name in ['table', 'room'] and 'need_chair' in self.data and 'need_table' in self.data:
            if self.data['need_chair'] == 'true' or self.data['need_table'] != helpdesk_fields.NEED_TABLE.NO:
                return sform.REQUIRED

        if self.editing_login_when_has_uid_or_guid(name):
            return sform.READONLY

        if self.editing_verified_phone(name):
            return sform.READONLY

        if not self.in_editable_state():
            return sform.READONLY

        if self.editing_by_recruiter():
            if name in ENABLED_FIELDS or name in ENABLED_FIELDS_FOR_RECRUITER:
                return super(EmployeeEditForm, self).get_field_state(name)

        if self.editing_join_at_by_adopter(name):
            return super(EmployeeEditForm, self).get_field_state(name)

        if self.in_not_editable_state_for_non_recruiter():
            return sform.READONLY

        if name not in ENABLED_FIELDS:
            return sform.READONLY

        return super(EmployeeEditForm, self).get_field_state(name)

    def clean(self):
        remove_readonly_fields(self)

        if 'login' in self.cleaned_data:
            validate_login_with_candidate_type(
                login=self.cleaned_data['login'],
                candidate_type=self.action_context.preprofile.candidate_type,
                preprofile_id=self.action_context.preprofile.id,
            )

        if self.get_field_state('hardware_profile') != sform.READONLY:
            department_url = getattr(self.cleaned_data.get('department'), 'url', None) or self.initial.get('department')
            hardware_profile = self.cleaned_data.get('hardware_profile')
            validate_hardware_profile_by_department(
                profile=hardware_profile,
                department_url=department_url,
                form=self,
            )
        validate_room_with_candidate_type(
            room=self.cleaned_data.get('room'),
            table=self.cleaned_data.get('table'),
            preprofile=self.action_context.preprofile,
            form=self,
        )

        return self.cleaned_data

    def get_photo(self, initial):
        if 'photo' in initial:
            return PreprofileAvatarCollection(
                owner=self.action_context.preprofile,
            ).get_link_for('is_main', True)
        return ''
