import sform

from staff.preprofile.forms.fields import (
    contacts_fields,
    helpdesk_fields,
    official_fields,
    personal_fields,
    technical_fields,
)
from staff.preprofile.forms.validation import validate_login_with_candidate_type
from staff.preprofile.models import (
    AUTOHIRE_HARDWARE_PROFILE_TYPE_TO_ID,
    NEED_TABLE,
    HardwareProfile,
)


def _get_autohire_hardware_profile(profile_type):
    if not profile_type:
        return None
    profile_id = AUTOHIRE_HARDWARE_PROFILE_TYPE_TO_ID[profile_type]
    return (
        HardwareProfile.objects
        .filter(
            profile_id=profile_id,
            departments__url='outstaff',
        )
        .first()
    )


class EmployeeForm(sform.SForm):
    login = official_fields.human_login()
    office = official_fields.office()
    position = official_fields.position(required=False)
    position_staff_text = official_fields.position_staff_text()
    join_at = official_fields.join_at()
    department = official_fields.department()
    organization = official_fields.organization()
    abc_services = official_fields.femida_abc_services(required=False)
    candidate_type = official_fields.candidate_type()
    employment_type = official_fields.employment_type()
    date_completion_internship = official_fields.date_completion_internship(readonly=False)

    first_name = personal_fields.first_name()
    last_name = personal_fields.last_name()
    first_name_en = personal_fields.first_name_en()
    last_name_en = personal_fields.last_name_en()
    middle_name = personal_fields.middle_name(required=False)
    gender = personal_fields.gender()
    birthday = personal_fields.birthday(required=False)
    photo = personal_fields.photo(required=False)

    phone = contacts_fields.phone(required=False)
    email = contacts_fields.email(required=False)
    address = contacts_fields.address(required=False)

    os = helpdesk_fields.os()
    lang_ui = helpdesk_fields.lang_ui(required=False)

    recruiter = technical_fields.recruiter()
    hr_ticket = technical_fields.hr_ticket(required=False)
    adopt_ticket = technical_fields.adopt_ticket(required=False)
    femida_offer_id = technical_fields.femida_offer_id()
    is_eds_phone_verified = technical_fields.is_eds_phone_verified(required=False)

    is_autohire = technical_fields.is_autohire(required=False)
    hardware_profile_type = helpdesk_fields.autohire_hardware_profile_type(required=False)

    need_internal_phone = helpdesk_fields.need_internal_phone_unfilled(required=False)
    need_sip_redirect = helpdesk_fields.need_sip_redirect_unfilled(required=False)

    def get_field_state(self, name):
        if name == 'hardware_profile_type' and self.cleaned_data.get('is_autohire'):
            return sform.REQUIRED
        return super().get_field_state(name)

    def clean(self):
        cleaned_data = self.cleaned_data

        required_fields = ['login', 'candidate_type']

        if any([field not in cleaned_data for field in required_fields]):
            return None

        login = cleaned_data['login']
        candidate_type = cleaned_data['candidate_type']

        validate_login_with_candidate_type(login, candidate_type)

        hardware_profile_type = cleaned_data.pop('hardware_profile_type', None)
        if cleaned_data.get('is_autohire'):
            cleaned_data['need_table'] = NEED_TABLE.NO
            cleaned_data['need_phone'] = False
            cleaned_data['hardware_profile'] = _get_autohire_hardware_profile(hardware_profile_type)

        return super(EmployeeForm, self).clean()
