import sform
from django.core.exceptions import ValidationError

from staff.departments.models import DepartmentRoles
from staff.lib.models.roles_chain import has_roles_for_department

from staff.preprofile.forms.fields import (
    contacts_fields,
    helpdesk_fields,
    official_fields,
    personal_fields,
    technical_fields,
)
from staff.preprofile.forms.validation import (
    validate_hardware_profile_by_department,
    validate_login_with_candidate_type,
)
from staff.preprofile.models import CANDIDATE_TYPE


class ExternalConsultantForm(sform.SForm):
    first_name_en = personal_fields.first_name_en()
    last_name_en = personal_fields.last_name_en()
    first_name = personal_fields.first_name()
    last_name = personal_fields.last_name()
    middle_name = personal_fields.middle_name(required=False)
    gender = personal_fields.gender()
    birthday = personal_fields.birthday(required=False)

    phone = contacts_fields.phone(required=True)
    email = contacts_fields.email(required=False)
    address = contacts_fields.address(required=False)

    login = official_fields.human_login()
    candidate_type = official_fields.candidate_type(
        without=[CANDIDATE_TYPE.EXTERNAL_EMPLOYEE, CANDIDATE_TYPE.CURRENT_EMPLOYEE],
    )
    department = official_fields.department(restrict_to_department_url='ext')

    lang_ui = helpdesk_fields.lang_ui()

    hardware_profile = helpdesk_fields.hardware_profile()
    hardware_profile_description = helpdesk_fields.hardware_profile_description(required=False)
    need_internal_phone = helpdesk_fields.need_internal_phone()
    need_sip_redirect = helpdesk_fields.need_sip_redirect()

    need_vpn = helpdesk_fields.need_vpn()

    uid = technical_fields.uid(readonly=True)
    guid = technical_fields.guid(readonly=True)
    created_at = technical_fields.created_at(readonly=True)
    modified_at = technical_fields.modified_at(readonly=True)
    status = technical_fields.status(readonly=True)
    chief = technical_fields.chief(readonly=True)

    @property
    def is_editing(self):
        return self.base_initial['action_context'].preprofile

    @property
    def requested_by(self):
        return self.base_initial['action_context'].requested_by

    def get_field_state(self, name):
        if self.is_editing:
            return sform.READONLY

        return super(ExternalConsultantForm, self).get_field_state(name)

    def clean_department(self, value):
        roles = [DepartmentRoles.DEPUTY.value, DepartmentRoles.CHIEF.value]

        has_permission = self.requested_by.user.has_perm('preprofile.can_create_external_by_api')
        has_role = has_roles_for_department(value, self.requested_by, roles)
        if has_role or has_permission:
            return value

        raise ValidationError('Wrong department', code='invalid_choice')

    def clean(self):
        cleaned_data = self.cleaned_data

        required_fields = ['login', 'candidate_type']

        if any([field not in cleaned_data for field in required_fields]):
            return None

        login = cleaned_data['login']
        candidate_type = cleaned_data['candidate_type']

        validate_login_with_candidate_type(login, candidate_type)

        department_url = getattr(self.cleaned_data.get('department'), 'url', None) or self.initial.get('department')
        hardware_profile = self.cleaned_data.get('hardware_profile')
        validate_hardware_profile_by_department(
            profile=hardware_profile,
            department_url=department_url,
            form=self,
        )

        return super(ExternalConsultantForm, self).clean()
