import sform
from django.core.exceptions import ValidationError

from staff.lib.models.mptt import is_ancestor
from staff.map.models import Office
from staff.person.models import Staff, Organization
from staff.departments.models import Department
from staff.groups.models import Group, GROUP_TYPE_CHOICES

from staff.preprofile.forms import validation
from staff.preprofile.forms.fields.field_factory import field_factory
from staff.preprofile.forms.fields.custom_fields import MultipleServicesSuggestField
from staff.preprofile.forms.fields.stripped_char_field import StrippedCharField
from staff.preprofile.login_validation import MAX_LOGIN_LENGTH, MAX_ROBOT_LOGIN_LENGTH, validate_login
from staff.preprofile.models import CANDIDATE_TYPE, EMPLOYMENT


def robot_login_validator(value):
    if not value.startswith('robot-'):
        raise ValidationError('Login for robot must starts with robot-', code='login_invalid_prefix')

    validate_login(value, True)


def zombie_login_validator(value):
    if not value.startswith('zomb-'):
        raise ValidationError('Login for robot must starts with zomb-', code='login_invalid_prefix')

    validate_login(value, True)


human_login = field_factory(sform.CharField, max_length=MAX_LOGIN_LENGTH)
robot_login = field_factory(sform.CharField, max_length=MAX_ROBOT_LOGIN_LENGTH, validators=[robot_login_validator])
zombie_login = field_factory(sform.CharField, max_length=MAX_ROBOT_LOGIN_LENGTH, validators=[zombie_login_validator])

employment_type = field_factory(sform.ChoiceField, choices=EMPLOYMENT.choices(), state=sform.REQUIRED)

join_at = field_factory(sform.DateField, validators=[validation.validate_date])

position = field_factory(StrippedCharField, max_length=150)
position_staff_text = field_factory(StrippedCharField, max_length=150)

abc_services = field_factory(
    MultipleServicesSuggestField,
    queryset=Group.objects.filter(intranet_status=1).filter(type=GROUP_TYPE_CHOICES.SERVICE),
    label_fields={'caption': ['name']},
    to_field_name='code',
)

femida_abc_services = field_factory(
    MultipleServicesSuggestField,
    queryset=Group.objects.filter(intranet_status=1).filter(type=GROUP_TYPE_CHOICES.SERVICE),
    label_fields={'caption': ['name']},
)

responsible = field_factory(
    sform.SuggestField,
    queryset=Staff.objects.filter(is_dismissed=False),
    to_field_name='login',
    label_fields=('first_name', 'last_name'),
)

organization = field_factory(
    sform.ModelChoiceField,
    queryset=Organization.objects.filter(intranet_status=1).order_by('name'),
    label_extractor='name',
)

office = field_factory(
    sform.ModelChoiceField,
    queryset=Office.objects.filter(intranet_status=1).order_by('name'),
    label_extractor='name',
)

date_completion_internship = field_factory(
    sform.DateField,
    validators=[validation.validate_date],
    required=False,
    readonly=True,
)


def department(restrict_to_department_url=None, required=True):
    if restrict_to_department_url and not isinstance(restrict_to_department_url, list):
        restrict_to_department_url = [restrict_to_department_url]

    def validator(dep):
        parent_deps = Department.objects.filter(url__in=restrict_to_department_url)
        for parent in parent_deps:
            if is_ancestor(parent, dep):
                return

        raise ValidationError('Wrong department', code='invalid_choice')

    validators = [validator] if restrict_to_department_url else []

    return sform.SuggestField(
        queryset=Department.objects.filter(intranet_status=1),
        to_field_name='url',
        label_fields='name',
        state=sform.REQUIRED if required else sform.NORMAL,
        validators=validators,
    )


def candidate_type(without=None, required=True):
    state = sform.REQUIRED if required else sform.NORMAL

    if not without:
        return sform.ChoiceField(choices=CANDIDATE_TYPE.choices(), state=state)

    if isinstance(without, list):
        choices = [c for c in CANDIDATE_TYPE.choices() if c[0] not in without]
    else:
        choices = [c for c in CANDIDATE_TYPE.choices() if without != c[0]]
    return sform.ChoiceField(choices=choices, state=state)
