import logging
import re
from datetime import date
import phonenumbers
from django.core.exceptions import ValidationError

from staff.person.models import Staff

from staff.preprofile.models.hardware_profile import ProfileForDepartment
from staff.preprofile.login_validation import validate_login, validate_by_uniqueness_in_preprofiles
from staff.preprofile.models import CANDIDATE_TYPE
from staff.preprofile.table_validation import validate_room, validate_table

logger = logging.getLogger(__name__)

VALID_LETTERS = 'a-zA-Zа-яА-ЯёЁҐґІіЇїЄєşŞıİçÇöÖüÜĞğÄäßĆćČčŁłŃńŚśŠšŹźŽžŬŭ'
VALID_LETTERS_EN = 'a-zA-Z'

name_regex = re.compile(
    r'^[{letter_character}][{letter_character}\-\'’ () ́]*$'.format(letter_character=VALID_LETTERS)
)

en_name_regex = re.compile(
    r'^[{letter_character}][{letter_character}\-\'’ () ́]*$'.format(
        letter_character=VALID_LETTERS_EN
    )
)

middle_name_regex = re.compile(
    r'^[{letter_character}][{letter_character}\-\'’ ́ ]*$'.format(
        letter_character=VALID_LETTERS
    )
)


def validate_value_is_not_empty(value):
    if not value:
        raise ValidationError('Should not be empty', code='empty_error')


def validate_date(value):
    if value and value < date(1900, 1, 1):
        raise ValidationError('Invalid date', code='invalid_date')


def _validate_name(value, regex, validation_error):
    value = value.strip()

    if '  ' in value:
        raise validation_error

    if '--' in value:
        raise validation_error

    if '\'\'' in value:
        raise validation_error

    if not regex.match(value):
        raise validation_error


def validate_name(value):
    _validate_name(value, name_regex, ValidationError('Invalid name', code='invalid_name'))


def validate_last_name(value):
    _validate_name(value, name_regex, ValidationError('Invalid name', code='invalid_last_name'))


def validate_name_en(value):
    _validate_name(value, en_name_regex, ValidationError('Invalid name', code='invalid_name'))


def validate_last_name_en(value):
    _validate_name(value, en_name_regex, ValidationError('Invalid name', code='invalid_last_name'))


def validate_middle_name(value):
    _validate_name(value, middle_name_regex, ValidationError('Invalid middle name', code='invalid_middle_name'))


def validate_dismissed_staff_exists(login):
    if not Staff.objects.filter(login=login, is_dismissed=True, memorial_profile__isnull=True).exists():
        raise ValidationError('There is no such employee', code='login_doesnt_exist')


def validate_staff_exists(login):
    if not Staff.objects.filter(login=login, is_dismissed=False).exists():
        raise ValidationError('There is no such employee', code='login_doesnt_exist')


def validate_phone(value, country_code='RU'):
    try:
        parsed = phonenumbers.parse(value, country_code)
    except phonenumbers.NumberParseException as ex:
        raise ValidationError(str(ex), code='invalid_phone_number')

    if not phonenumbers.is_valid_number(parsed):
        raise ValidationError('Phone number is invalid', code='invalid_phone_number')

    if not phonenumbers.is_possible_number(parsed):
        raise ValidationError('Phone number is impossible', code='invalid_phone_number')


def validate_login_with_candidate_type(login, candidate_type, preprofile_id=None):
    if candidate_type == CANDIDATE_TYPE.NEW_EMPLOYEE:
        validate_login(login, False, preprofile_id)
    elif candidate_type == CANDIDATE_TYPE.FORMER_EMPLOYEE:
        validate_dismissed_staff_exists(login)
        validate_by_uniqueness_in_preprofiles(login, preprofile_id)
    elif candidate_type in (CANDIDATE_TYPE.EXTERNAL_EMPLOYEE, CANDIDATE_TYPE.CURRENT_EMPLOYEE):
        validate_staff_exists(login)
        validate_by_uniqueness_in_preprofiles(login, preprofile_id)
    else:
        raise ValidationError('Not supported', code='invalid_candidate_type')


def validate_hardware_profile_by_department(profile, department_url, form):
    if not department_url:
        return
    if not profile and not ProfileForDepartment.objects.filter(department__url=department_url).exists():
        return
    if profile and ProfileForDepartment.objects.filter(department__url=department_url, profile=profile).exists():
        return
    form._errors[('hardware_profile',)] = [ValidationError('Invalid hardware profile', code='invalid_choice')]


def validate_room_with_candidate_type(room, table, preprofile, form):
    try:
        if room is not None:
            validate_room(room, preprofile)
    except ValidationError as err:
        form._errors[('room',)] = [err]

    try:
        if table is not None:
            validate_table(table, preprofile)
    except ValidationError as err:
        form._errors[('table',)] = [err]
