import re
import logging
from itertools import chain

import ldap
from django.core.exceptions import ValidationError
from django.db.models import Q, F, Value, CharField

from staff.person.models import Staff

from staff.lib.utils.ordered_choices import OrderedChoices
from staff.maillists.models import List

from staff.preprofile.ldap_utils import get_guid_from_ldap
from staff.preprofile.models import IllegalLogin, Preprofile, PREPROFILE_STATUS
from staff.preprofile.utils import get_ip


logger = logging.getLogger(__name__)


MIN_LOGIN_LENGTH = 4
MAX_LOGIN_LENGTH = 13
MAX_ROBOT_LOGIN_LENGTH = 20

ALLOWED_CHAR_REGEXP = r'[a-z0-9\-]'
LOGIN_BEGINS_WITH_REGEXP = r'^[a-z]'

LOGIN_VALIDATION_ERROR = OrderedChoices(
    ('BAD_LENGTH', 'login_bad_length'),
    ('BAD_SYMBOLS', 'login_bad_symbols'),
    ('LOGIN_IN_BLACKLIST', 'login_in_blacklist'),
    ('LOGIN_NOT_UNIQUE', 'login_not_unique'),
    ('LOGIN_LDAP_CONFLICT', 'login_ldap_conflict'),
    ('LOGIN_LDAP_UNAVAILABLE', 'login_ldap_unavailable'),
    ('LOGIN_LIKENESS', 'login_likeness'),
    ('LOGIN_MAILLIST_CONFLICT', 'login_maillist_conflict'),
    ('LOGIN_DNS_CONFLICT', 'login_dns_conflict'),
)


def validate_login(login, is_robot, preprofile_id=None):
    validate_login_length(login, is_robot)
    validate_login_syntax(login)
    validate_for_illegal_login(login)
    validate_by_uniqueness(login)
    validate_by_uniqueness_in_preprofiles(login, preprofile_id)
    validate_in_ldap(login)

    if not is_robot:
        validate_likeness_in_staff(login)
        validate_likeness_in_preprofiles(login, preprofile_id)

    validate_for_maillist(login)
    validate_for_dns(login)


def validate_login_length(login, is_robot):
    login_length = len(login)
    if login_length < MIN_LOGIN_LENGTH:
        raise ValidationError('Login too short', code=LOGIN_VALIDATION_ERROR.BAD_LENGTH)

    max_length = MAX_ROBOT_LOGIN_LENGTH if is_robot else MAX_LOGIN_LENGTH

    if login_length > max_length:
        raise ValidationError('Login too long', code=LOGIN_VALIDATION_ERROR.BAD_LENGTH)


def validate_login_syntax(login):
    if not re.match(r'^' + ALLOWED_CHAR_REGEXP + r'*$', login):
        raise ValidationError('Login contains wrong characters', code=LOGIN_VALIDATION_ERROR.BAD_SYMBOLS)

    if not re.match(LOGIN_BEGINS_WITH_REGEXP, login):
        raise ValidationError('Login should start with a letter', code=LOGIN_VALIDATION_ERROR.BAD_SYMBOLS)

    if login.isdigit():
        raise ValidationError('Login cannot be a number', code=LOGIN_VALIDATION_ERROR.BAD_SYMBOLS)

    if '--' in login:
        raise ValidationError('Login should not have double dash', code=LOGIN_VALIDATION_ERROR.BAD_SYMBOLS)

    if login.endswith('-'):
        raise ValidationError('Login should not starts or ends with dash', code=LOGIN_VALIDATION_ERROR.BAD_SYMBOLS)

    if login.endswith('-api'):
        raise ValidationError('Login should not contain `-api`', code=LOGIN_VALIDATION_ERROR.BAD_SYMBOLS)


def validate_for_illegal_login(login):
    lookups = Q(login=login, exact_match=True) | Q(given_login__icontains=F('login'), exact_match=False)
    illegal_login_exists = (
        IllegalLogin.objects
        .annotate(given_login=Value(login, output_field=CharField()))
        .filter(lookups)
        .exists()
    )
    if illegal_login_exists:
        raise ValidationError('Login is in blacklist', code=LOGIN_VALIDATION_ERROR.LOGIN_IN_BLACKLIST)


def validate_by_uniqueness(login):
    if Staff.objects.filter(login=login).exists():
        raise ValidationError('Login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_NOT_UNIQUE)


def validate_by_uniqueness_in_preprofiles(login, exclude_preprofile_id=None):
    exclude_preprofile_id = exclude_preprofile_id if exclude_preprofile_id else -1

    terminal_statuses = [
        PREPROFILE_STATUS.CLOSED,
        PREPROFILE_STATUS.CANCELLED,
    ]

    qs = (
        Preprofile.objects
        .exclude(status__in=terminal_statuses)
        .exclude(id=exclude_preprofile_id)
        .filter(login=login)
        .exists()
    )

    if qs:
        raise ValidationError('Login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_NOT_UNIQUE)


def validate_in_ldap(login):
    try:
        if get_guid_from_ldap(login):
            raise ValidationError('Login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_LDAP_CONFLICT)
    except ldap.LDAPError:
        logger.exception('Error while getting login info from AD for %s', login)
        raise ValidationError('Login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_LDAP_UNAVAILABLE)


def _add_line_start_and_line_end(regexp):
    return r'^' + regexp + r'$'


def _gen_replaces_and_removals(login):
    for i, char in enumerate(login):
        yield login[:i], _add_line_start_and_line_end(login[:i] + ALLOWED_CHAR_REGEXP + r'?' + login[i+1:])


def _gen_inserts(login):
    for i, char in enumerate(login):
        yield login[:i], _add_line_start_and_line_end(login[:i] + ALLOWED_CHAR_REGEXP + login[i:])
    yield login, _add_line_start_and_line_end(login + ALLOWED_CHAR_REGEXP)


def _gen_swaps(login):
    for i in range(1, len(login)):
        yield login[:i-1] + login[i] + login[i-1] + login[i+1:]


def _likeness_qs(login):
    replaces_regexps = set(_gen_replaces_and_removals(login))
    inserts_regexps = set(_gen_inserts(login))

    qs = Q(login__in=set(_gen_swaps(login)))

    for startswith, regex in chain(replaces_regexps, inserts_regexps):
        qs = qs | (Q(login__startswith=startswith) & Q(login__iregex=regex))

    return qs


def validate_likeness_in_staff(login):
    if Staff.objects.filter(_likeness_qs(login)).exists():
        raise ValidationError('Similar login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_LIKENESS)


def validate_likeness_in_preprofiles(login, exclude_preprofile_id=None):
    exclude_preprofile_id = exclude_preprofile_id if exclude_preprofile_id else -1

    qs = (
        Preprofile.objects
        .exclude(status__in=[PREPROFILE_STATUS.CLOSED, PREPROFILE_STATUS.CANCELLED])
        .exclude(id=exclude_preprofile_id)
        .filter(_likeness_qs(login))
        .values_list('login', flat=True)
    )

    if qs.exists():
        raise ValidationError('Similar login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_LIKENESS)


def validate_for_maillist(login):
    # for mail lists dot and dash are the same symbols
    regex = re.sub(r'[-.]', '[-.]', login)
    if List.objects.filter(name__iregex=regex).exists():
        raise ValidationError('Login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_MAILLIST_CONFLICT)


def validate_for_dns(login):
    if login == 'any':
        raise ValidationError('Login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_DNS_CONFLICT)

    exclusions = {ip for item in ['any.yandex.ru'] for ip in get_ip(item)}
    domains_to_check = ['yandex.ru', 'yandex.net', 'yandex-team.ru']

    for domain in domains_to_check:
        name = '{login}.{domain}'.format(login=login, domain=domain)
        answers = get_ip(name)
        answers = [i for i in answers if i not in exclusions]
        if answers:
            raise ValidationError('Login already in use', code=LOGIN_VALIDATION_ERROR.LOGIN_DNS_CONFLICT)
