from django.conf import settings
from django.db import models

from staff.lib.utils.ordered_choices import OrderedChoices
from staff.map.models import Office, Room, Table
from staff.departments.models import Department
from staff.person.models import Staff, Organization, GENDER, LANG, EDU_STATUS, EDU_DIRECTION, EMPLOYMENT
from staff.preprofile.models.hardware_profile import HardwareProfile


NEED_TABLE = OrderedChoices(
    ('NO', 'NO', 'no'),
    ('YES', 'YES', 'yes'),
    ('ALREADY_HAVE', 'ALREADY_HAVE', 'already_have'),
)

PREPROFILE_STATUS = OrderedChoices(
    'NEW',  # новая
    'PREPARED',  # Заполнена кандидатом
    'APPROVED',  # подтверждена руководителем
    'READY',  # подготовка рабочего места
    'CLOSED',  # добавлена на стафф
    'CANCELLED',  # отменена
)

OS_TYPE = OrderedChoices(
    'UBUNTU',
    'KUBUNTU',
    'WINDOWS',
    'MAC',
    'NO_OS',
)

CITIZENSHIP = OrderedChoices(
    'RUSSIAN',
    'BELORUSSIAN',
    'UKRAINIAN',
    'KAZAKHSTAN',
    'KYRGYZSTAN',
    'ARMENIA',
    'OTHER',
)
CITIZENSHIP_TRANSLATE = {
    'российское': CITIZENSHIP.RUSSIAN,
    'белорусское': CITIZENSHIP.BELORUSSIAN,
    'украинское': CITIZENSHIP.UKRAINIAN,
    'казахстанское': CITIZENSHIP.KAZAKHSTAN,
    'казахское': CITIZENSHIP.KAZAKHSTAN,
    'киргизское': CITIZENSHIP.KYRGYZSTAN,
    'армянское': CITIZENSHIP.ARMENIA,
    'другое': CITIZENSHIP.OTHER,
}
CITIZENSHIP_TRANSLATE.update({it.lower(): it for it in CITIZENSHIP_TRANSLATE.values()})

EMAIL_DOMAIN = OrderedChoices(
    ('YANDEX_TEAM_RU', 'yandex-team.ru'),
    ('YANDEX_TEAM_COM_UA', 'yandex-team.com.ua'),
    ('YANDEX_TEAM_COM', 'yandex-team.com'),
    ('YANDEX_TEAM_COM_TR', 'yandex-team.com.tr'),
    ('YAMONEY_RU', 'yamoney.ru'),
    ('YAPROBKI_RU', 'yaprobki.ru'),
)

FORM_TYPE = OrderedChoices(
    ('EMPLOYEE', 'employee'),
    ('ROTATION', 'rotation'),
    ('MONEY', 'money'),
    ('ROBOT', 'robot'),
    ('ZOMBIE', 'zombie'),
    ('EXTERNAL', 'external'),
    ('OUTSTAFF', 'outstaff'),
    ('MASS_HIRE', 'masshire'),
)

CANDIDATE_TYPE = OrderedChoices(
    ('NEW_EMPLOYEE', 'NEW_EMPLOYEE', 'new_employee'),
    ('FORMER_EMPLOYEE', 'FORMER_EMPLOYEE', 'former_employee'),
    ('EXTERNAL_EMPLOYEE', 'EXTERNAL_EMPLOYEE', 'external_employee'),
    ('CURRENT_EMPLOYEE', 'CURRENT_EMPLOYEE', 'current_employee'),
)

# Явно зафиксированные профили оборудования для автонайма
AUTOHIRE_HARDWARE_PROFILE_TYPE = OrderedChoices(
    ('NO', 'no'),
    ('TOKEN', 'token'),
    ('LAPTOP', 'laptop'),
)
AUTOHIRE_HARDWARE_PROFILE_TYPE_TO_ID = {
    AUTOHIRE_HARDWARE_PROFILE_TYPE.NO: '110',
    AUTOHIRE_HARDWARE_PROFILE_TYPE.TOKEN: '109',
    AUTOHIRE_HARDWARE_PROFILE_TYPE.LAPTOP: '108',
}

ROOT_DEPARTMENT = OrderedChoices(
    ('YANDEX', settings.YANDEX_DEPARTMENT_ID, 'yandex'),
    ('EXTERNAL', settings.EXT_DEPARTMENT_ID, 'external'),
    ('OUTSTAFF', settings.OUTSTAFF_DEPARTMENT_ID, 'outstaff'),
)
INTERNSHIP_FULLTIME = OrderedChoices(
    ('INTERNSHIP', 'internship'),
    ('FULLTIME', 'fulltime'),
)


class Preprofile(models.Model):
    form_type = models.CharField(max_length=10, choices=FORM_TYPE.choices(), blank=False)
    status = models.CharField(max_length=15, choices=PREPROFILE_STATUS.choices(), blank=False)
    candidate_type = models.CharField(max_length=25, choices=CANDIDATE_TYPE.choices(), blank=False)
    employment_type = models.CharField(max_length=1, choices=EMPLOYMENT.choices(), null=True, blank=False)

    department = models.ForeignKey(Department)
    organization = models.ForeignKey(Organization, null=True, on_delete=models.SET_NULL)

    abc_services = models.ManyToManyField('django_intranet_stuff.Group', through='PreprofileABCServices')

    office = models.ForeignKey(Office, null=True, related_name='+', on_delete=models.SET_NULL)
    position_staff_text = models.CharField(max_length=150)
    position = models.CharField(max_length=150, null=True)
    docs_and_hw_address = models.CharField(max_length=100)
    hardware_office = models.ForeignKey(Office, null=True, related_name='+', on_delete=models.SET_NULL)

    photo = models.CharField(max_length=255, null=True)

    first_name = models.CharField(max_length=30)
    first_name_en = models.CharField(max_length=50)
    middle_name = models.CharField(max_length=50, null=True)
    last_name = models.CharField(max_length=30)
    last_name_en = models.CharField(max_length=100)

    join_at = models.DateField(db_index=True)
    birthday = models.DateField(null=True)
    citizenship = models.CharField(max_length=16, choices=CITIZENSHIP.choices(), null=True, blank=True)

    login = models.CharField(max_length=50, null=True)
    phone = models.CharField(max_length=100, null=True)
    email = models.CharField(max_length=100, null=True)
    address = models.CharField(max_length=1024, null=True)
    gender = models.CharField(max_length=1, choices=GENDER.choices(), blank=False)
    ext_yndx_login = models.CharField(max_length=50, null=True)
    os = models.CharField(max_length=15, choices=OS_TYPE.choices(), null=True, blank=False)

    # Поля education_* больше не используются, но исторические данные в БД пока оставляем
    education_status = models.CharField(max_length=1, choices=EDU_STATUS.choices(), null=True, blank=False)
    education_direction = models.CharField(max_length=1, choices=EDU_DIRECTION.choices(), null=True, blank=False)
    education_place = models.CharField(max_length=255, null=True)
    education_date = models.DateField(null=True)

    crm_alias = models.CharField(max_length=50, null=True)
    email_domain = models.CharField(
        max_length=20,
        choices=EMAIL_DOMAIN.choices(),
        default=EMAIL_DOMAIN.YANDEX_TEAM_RU,
    )
    need_vpn = models.NullBooleanField(null=True)
    lang_ui = models.CharField(max_length=2, choices=LANG.choices(), null=True, blank=False, default=LANG.RU)

    need_table = models.CharField(max_length=15, choices=NEED_TABLE.choices(), null=True, blank=False)
    need_chair = models.NullBooleanField(null=True)
    need_phone = models.NullBooleanField(null=True)

    mentor = models.ForeignKey(Staff, related_name='+', null=True, on_delete=models.SET_NULL)

    hardware_profile = models.ForeignKey(HardwareProfile, null=True, on_delete=models.SET_NULL)
    hardware_profile_description = models.CharField(max_length=1000, null=True)
    hardware_profile_needs_table = models.NullBooleanField(null=True)

    table = models.ForeignKey(Table, null=True, on_delete=models.SET_NULL)
    room = models.ForeignKey(Room, null=True, on_delete=models.SET_NULL)

    robot_owner = models.ForeignKey(Staff, related_name='+', null=True)
    zombie_hw = models.TextField(null=True)
    remote_access = models.TextField(null=True)
    network_connected = models.NullBooleanField()
    need_tv = models.NullBooleanField()

    need_internal_phone = models.NullBooleanField(null=True)
    need_sip_redirect = models.NullBooleanField(null=True)

    femida_offer_id = models.IntegerField(null=True, unique=True, blank=False)
    recruiter = models.ForeignKey(Staff, null=False, related_name='+', db_index=True, on_delete=models.PROTECT)
    uid = models.CharField(max_length=16, null=True, blank=False)
    guid = models.CharField(max_length=47, null=True, blank=False)
    approved_by = models.ForeignKey(Staff, related_name='+', null=True, on_delete=models.SET_NULL)
    adopted_by = models.ForeignKey(Staff, related_name='+', null=True, on_delete=models.SET_NULL)

    hr_ticket = models.CharField(max_length=15, null=True)
    adopt_ticket = models.CharField(max_length=15, null=True)
    hdrfs_ticket = models.CharField(max_length=15, null=True)
    supply_ticket = models.CharField(max_length=15, null=True)
    tools_ticket = models.CharField(max_length=15, null=True)

    duties = models.CharField(max_length=1024, default='')
    duties_en = models.CharField(max_length=1024, default='')

    ext_form_link = models.CharField(max_length=128, default='')

    masshire_tag = models.CharField(max_length=128, null=True, db_index=True)

    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)

    date_completion_internship = models.DateField(null=True, default=None)

    is_eds_phone_verified = models.NullBooleanField()
    is_autohire = models.NullBooleanField()

    class Meta:
        permissions = (
            ('can_view_all_preprofiles', 'Can see all preprofiles'),
            ('available_preprofile_for_externals',
             'External subordinates can see pages preprofile/*'),
            ('can_create_yamoney', 'Can create Ya.Money preprofiles'),
            ('can_outstaff', 'Can create outstaff'),
            ('chief_that_can_adopt', 'Special CHIEF that can adopt'),
            ('can_manage_masshire', 'Can manage mass hire processes'),
            ('can_see_rotation_history', 'Can see rotation history'),
            ('can_create_outstaff_by_api', 'Can create outstaff preprofile by API'),
            ('can_create_external_by_api', 'Can create external preprofile by API'),
            ('can_create_robot_by_api', 'Can create robot preprofile by API'),
            ('can_create_zombie_by_api', 'Can create zombie preprofile by API'),
            (
                'available_preprofile_new_form_api_for_externals',
                'Externals can access preprofile API',
            ),
        )

    def __str__(self):
        return '{s.login} ({s.status})'.format(s=self)

    @property
    def is_robot_or_zombie(self):
        return self.form_type in [FORM_TYPE.ROBOT, FORM_TYPE.ZOMBIE]

    @property
    def has_uid_or_guid(self):
        return self.uid or self.guid


class PreprofileABCServices(models.Model):
    group = models.ForeignKey('django_intranet_stuff.Group')
    preprofile = models.ForeignKey('Preprofile')

    class Meta:
        unique_together = (('group', 'preprofile'),)
