from django.conf import settings
from staff.django_intranet_notifications import Notification
from staff.person.models import Staff

from staff.preprofile.models import (
    CANDIDATE_TYPE,
    FORM_TYPE,
    NEED_TABLE,
)

from staff.lib.models.roles_chain import direct_chief_for_department


class PreprofileNotification(Notification):

    def __init__(self, preprofile, *args, **kwargs):
        context = kwargs.setdefault('context', {})
        context['preprofile'] = preprofile
        if 'department' not in kwargs:
            kwargs['department'] = preprofile.department
        if 'office' not in kwargs:
            kwargs['office'] = preprofile.office
        kwargs['template'] = self.get_template()
        super(PreprofileNotification, self).__init__(*args, **kwargs)

    def get_subj_id(self):
        return 'preprofile_%d' % self.context['preprofile'].id

    def send(self, **params):
        if settings.SEND_PREPROFILE_NOTIFICATIONS:
            return super(PreprofileNotification, self).send(**params)

    @classmethod
    def get_template(cls):
        return 'notifications/{}.html'.format(cls.template)


class AskToFillPreprofile(PreprofileNotification):
    template = 'ask_to_fill_preprofile'


class HelpRequest(PreprofileNotification):
    template = 'help_request'


class HelpD7Request(PreprofileNotification):
    template = 'helpd7_request'


class OfferCancelled(PreprofileNotification):
    template = 'offer_cancelled'


class PreprofileApproved(PreprofileNotification):
    template = 'preprofile_approved'


class PersonAdopted(PreprofileNotification):
    template = 'person_adopted'


class PersonForgotten(PreprofileNotification):
    template = 'person_forgotten'


class PreprofileChanged(PreprofileNotification):
    template = 'preprofile_changed'

    def __init__(self, notify_about, *args, **kwargs):
        context = kwargs.setdefault('context', {})
        context.update(notify_about)
        super(PreprofileChanged, self).__init__(*args, **kwargs)


class PreprofilePrepared(PreprofileNotification):
    template = 'preprofile_prepared'


class ProposeLearning(PreprofileNotification):
    template = 'propose_learning'


class RemindAboutPreprofile(PreprofileNotification):
    template = 'remind_about_preprofile'

    def __init__(self, is_urgent=False, *args, **kwargs):
        context = kwargs.setdefault('context', {})
        context['is_urgent'] = is_urgent
        super(RemindAboutPreprofile, self).__init__(*args, **kwargs)


NOTIFICATION_TYPES = dict(
    ask_to_fill_preprofile=AskToFillPreprofile,
    help_request=HelpRequest,
    helpd7_request=HelpD7Request,
    offer_cancelled=OfferCancelled,
    preprofile_approved=PreprofileApproved,
    person_adopted=PersonAdopted,
    person_forgotten=PersonForgotten,
    preprofile_changed=PreprofileChanged,
    preprofile_prepared=PreprofilePrepared,
    propose_learning=ProposeLearning,
    remind_about_preprofile=RemindAboutPreprofile,
)


def _get_chief_email(preprofile):
    chief = direct_chief_for_department(preprofile.department, fields=['work_email'])
    return chief.get('work_email') if chief else None


def help_request(preprofile):
    staff_robot = Staff.objects.get(login=settings.ROBOT_STAFF_LOGIN)

    if preprofile.need_chair or preprofile.need_table == NEED_TABLE.YES:
        # Only if help is needed
        ticket = HelpD7Request(
            preprofile=preprofile,
            target='STARTREK_ISSUE',
        ).send(
            queue=settings.SUPPLY_QUEUE_ID,
            author=staff_robot,
        )
        preprofile.supply_ticket = ticket['key'] if ticket else None
    preprofile.save()


def notify_preprofile_adopted(preprofile):
    if preprofile.form_type in (FORM_TYPE.MONEY, FORM_TYPE.EXTERNAL, FORM_TYPE.OUTSTAFF):
        email = _get_chief_email(preprofile)

        if email:
            PersonAdopted(
                preprofile=preprofile,
                target='@',
            ).send(recipients=[email])
    elif preprofile.form_type == FORM_TYPE.EMPLOYEE:
        email = _get_chief_email(preprofile)

        if email:
            ProposeLearning(
                preprofile=preprofile,
                target='@',
            ).send(recipients=[email])


def notify_preprofile_cancelled(preprofile):
    if preprofile.form_type not in (FORM_TYPE.EMPLOYEE, FORM_TYPE.OUTSTAFF):
        return
    email = _get_chief_email(preprofile)

    if email:
        OfferCancelled(
            preprofile=preprofile,
            target='@',
        ).send(recipients=[email])


def notify_preprofile_prepared(preprofile):
    if preprofile.form_type != FORM_TYPE.OUTSTAFF:
        return
    emails = [_get_chief_email(preprofile)]
    if preprofile.recruiter:
        emails.append(preprofile.recruiter.email)
    PreprofilePrepared(
        preprofile=preprofile,
        target='@',
    ).send(recipients=[_f for _f in emails if _f])


def notify_preprofile_approved(preprofile):
    is_yamoney = preprofile.form_type == FORM_TYPE.MONEY
    is_external = preprofile.candidate_type == CANDIDATE_TYPE.EXTERNAL_EMPLOYEE
    if is_yamoney and is_external:
        return
    help_request(preprofile)
    if preprofile.form_type not in (FORM_TYPE.EMPLOYEE, FORM_TYPE.OUTSTAFF):
        return
    emails = [_get_chief_email(preprofile)]
    if preprofile.recruiter:
        emails.append(preprofile.recruiter.email)
    mails_to_send = [it for it in emails if it and it != preprofile.approved_by.email]
    PreprofileApproved(
        preprofile=preprofile,
        target='preprofile_approved',
    ).send(recipients=mails_to_send)


def notify_preprofile_created(preprofile):
    if preprofile.form_type != FORM_TYPE.EMPLOYEE:
        return
    email = _get_chief_email(preprofile)
    if not email:
        return
    AskToFillPreprofile(
        preprofile=preprofile,
        target='@',
    ).send(recipients=[email])


def notify_preprofile_changed(preprofile, changed):
    notify_about = {
        field: info['new'] if isinstance(info, dict) else info
        for field, info in changed.items()
        if field in ('join_at', 'login', 'department')
    }
    if not notify_about:
        return
    emails = []
    if preprofile.recruiter:
        emails.append(preprofile.recruiter.email)
    if 'department' in notify_about:
        emails.append(_get_chief_email(preprofile))
    PreprofileChanged(
        preprofile=preprofile,
        notify_about=notify_about,
        target='preprofile_changed',
    ).send(recipients=[_f for _f in emails if _f])
