import datetime
import logging
import operator
from collections import defaultdict
from functools import reduce
from typing import List

from django.conf import settings
from django.db.models import Q

from staff.celery_app import app
from staff.lib.db import atomic
from staff.lib.log import log_context
from staff.lib.models import departments_chain
from staff.lib.sync_tools.diff_merger import DataDiffMerger
from staff.lib.sync_tools.updater import Updater
from staff.lib.tasks import LockedTask
from staff.person.models import (
    Staff,
    StaffPhone,
    VerifyCode,
    VERIFY_CODE_OBJECT_TYPES,
    VERIFY_STATE,
)
from staff.person_profile.controllers.digital_sign import create_digital_sign, prepare_digital_sign

from staff.preprofile.controllers import ControllerError, AccessDeniedControllerError
from staff.preprofile.models import (
    HardwareProfile,
    ProfileForDepartment,
    PersonAdoptApplication,
    Preprofile,
    PREPROFILE_STATUS,
    FORM_TYPE,
    STATUS,
)
from staff.preprofile.notifications import PersonForgotten, RemindAboutPreprofile
from staff.preprofile.reminder_email_gatherer import gather_remind_emails
from staff.preprofile.repository import Repository
from staff.preprofile.sync_hardware_profiles import ProfilesDataGenerator, ProfilesDataSource
from staff.preprofile.utils import try_create_masshire_preprofiles_by_parsed_data, normalize_phone


logger = logging.getLogger(__name__)


@app.task
class ProcessAdoptApplications(LockedTask):
    def locked_run(self, *args, **kwargs):
        qs = PersonAdoptApplication.objects.filter(status__in=[STATUS.NEW, STATUS.FAILED])

        for application in qs:
            with atomic():
                self._process_application(application)

    def _process_application(self, application):
        """
        :type application: ProcessAdoptApplications
        """
        logging.info('Processing preprofile %s', application.preprofile_id)

        controller = Repository(application.adopter).existing(preprofile_id=application.preprofile_id)

        if controller.status in [PREPROFILE_STATUS.CLOSED, PREPROFILE_STATUS.CANCELLED]:
            logging.info(
                'Preprofile %s already closed or cancelled, marking it DONE',
                application.preprofile_id,
            )
            application.status = STATUS.DONE
            application.save()
            return

        logging.info('Preprofile %s is not adopted, adopting it', application.preprofile_id)

        try:
            controller.adopt()
        except ControllerError as e:
            logging.exception(
                'Preprofile %s was not adopted, errors %s',
                application.preprofile_id,
                e.errors_dict,
            )
            return
        except AccessDeniedControllerError:
            logging.exception(
                'Preprofile %s was not adopted, user %s has no rights to adopt it',
                application.preprofile_id,
                application.adopter.login,
            )
            return
        except Exception:
            logging.exception('Preprofile %s was not adopted')
            return

        application.status = STATUS.DONE
        application.save()


@app.task
class UpdatePassportAndGuid(LockedTask):

    def locked_run(self, *args, **kwargs):
        for ctrl in Repository(Staff.objects.get(login=settings.ROBOT_STAFF_LOGIN)).existing_approved():
            ctrl.try_make_ready()


@app.task
class SendNotifies(LockedTask):
    SIMPLE_REMIND_DAYS_LEFT = 11
    URGENT_REMIND_DAYS_LEFT = 8

    def _create_notification(self, preprofile, is_forgotten, is_urgent):
        if is_forgotten:
            return PersonForgotten(preprofile=preprofile, target='@')

        return RemindAboutPreprofile(preprofile=preprofile, target='@', is_urgent=is_urgent)

    def locked_run(self, *args, **kwargs):
        today = datetime.date.today()
        robot_staff = Staff.objects.get(login=settings.ROBOT_STAFF_LOGIN)
        preprofiles_to_remind = (
            Repository(robot_staff)
            .preprofiles_for_reminder(self.SIMPLE_REMIND_DAYS_LEFT, self.URGENT_REMIND_DAYS_LEFT)
        )
        emails_for_preprofiles = gather_remind_emails(preprofiles_to_remind)
        result = []

        for preprofile in preprofiles_to_remind:
            with log_context(preprofile_id=preprofile.id):
                is_urgent = preprofile.join_at == today + datetime.timedelta(days=self.URGENT_REMIND_DAYS_LEFT)
                is_forgotten = preprofile.join_at <= today

                notification = self._create_notification(preprofile, is_forgotten, is_urgent)
                emails = emails_for_preprofiles[preprofile.id].emails

                if is_urgent:
                    emails.append(emails_for_preprofiles[preprofile.id].urgent_emails)

                if emails:
                    result.append(notification.send(recipients=emails))

        return result


@app.task
class AutoAdoptPreprofiles(LockedTask):
    autoadopt_form_types = [
        FORM_TYPE.ROBOT,
        FORM_TYPE.ZOMBIE,
        FORM_TYPE.MONEY,
        FORM_TYPE.EXTERNAL,
    ]

    def locked_run(self, *args, **kwargs):
        autoadopt_q = (
            Q(form_type__in=self.autoadopt_form_types) |
            Q(
                form_type=FORM_TYPE.ROTATION,
                join_at__lte=datetime.date.today(),
            )
        )
        qs = (
            Preprofile.objects
            .filter(status=PREPROFILE_STATUS.READY)
            .filter(autoadopt_q)
        )
        preprofile_id = kwargs.get('preprofile_id')
        if preprofile_id:
            qs = qs.filter(id=preprofile_id)

        robot_staff = Staff.objects.get(login=settings.ROBOT_STAFF_LOGIN)

        for preprofile in qs:
            try:
                with atomic():
                    controller = Repository(robot_staff).existing(preprofile.id)

                    if controller.can_be_adopted():
                        controller.adopt()
                    else:
                        logger.info(
                            'Skip adoption for preprofile %s with login %s as it\'s cannot be adopted now',
                            preprofile.id,
                            preprofile.login,
                        )
            except Exception:
                logger.exception('Exception while processing preprofile %s', preprofile.id)


@app.task
class SyncHardwareProfiles(LockedTask):
    def _create_updater(self) -> Updater:
        data_source = ProfilesDataSource(settings)
        data_generator = ProfilesDataGenerator(data_source)
        data_diff_merger = DataDiffMerger(data_generator)
        updater = Updater(data_diff_merger, logger)
        updater.source_type = 'Hardware profiles'
        return updater

    def locked_run(self, *args, **kwargs):
        self._create_updater().run_sync()

        department_ancestors = departments_chain.get_departments_tree()
        profiles = HardwareProfile.objects.values('id', 'url')

        profile_by_url = defaultdict(list)
        for profile in profiles:
            profile_by_url[profile['url']].append(profile['id'])

        new_profiles = []
        for department, ancestors in department_ancestors.items():
            order = 0
            for ancestor in ancestors:
                profile_ids = profile_by_url.get(ancestor['url'], [])
                for profile_id in profile_ids:
                    profile = ProfileForDepartment(department_id=department, profile_id=profile_id, order=order)
                    new_profiles.append(profile)
                    order += 1

        ProfileForDepartment.objects.all().delete()
        ProfileForDepartment.objects.bulk_create(new_profiles)


@app.task(max_retries=3)
class PrepareDigitalSigns(LockedTask):
    """
    STAFF-13427: Предварительно выпускаем ЭП за день до выхода
    """
    def locked_run(self, *args, **kwargs):
        types = VERIFY_CODE_OBJECT_TYPES
        initiator = Staff.objects.get(login='robot-femida').user

        tomorrow = datetime.date.today() + datetime.timedelta(days=1)
        preprofiles = (
            Preprofile.objects
            .filter(
                status=PREPROFILE_STATUS.READY,
                is_eds_phone_verified=True,
                join_at=tomorrow,
            )
            .values_list('id', 'login', 'femida_offer_id')
        )
        preprofiles_by_type = defaultdict(dict)
        for preprofile_id, login, femida_offer_id in preprofiles:
            object_type = types.OFFER if femida_offer_id else types.PREPROFILE
            object_id = femida_offer_id or preprofile_id
            preprofiles_by_type[object_type][object_id] = login

        if not preprofiles_by_type:
            return

        query = reduce(
            operator.or_,
            (Q(object_type=k, object_id__in=v) for k, v in preprofiles_by_type.items()),
            Q(pk=None),
        )
        verify_codes = (
            VerifyCode.objects
            .filter(state=VERIFY_STATE.VERIFIED)
            .filter(query)
        )
        for code in verify_codes:
            login = preprofiles_by_type[code.object_type][code.object_id]
            prepared, oebs_resp = prepare_digital_sign(code, initiator, login, code.phone_number)
            if not prepared:
                prepare_digital_sign_task.delay(code.id, login)
                if oebs_resp:
                    logger.warning(f'Digital sign preparing failed for {code}: OEBS responded with {oebs_resp}')
                else:
                    logger.warning(f'Digital sign preparing failed for {code}.')


@app.task
def create_masshire_preprofiles(recruiter_login: str, data: list):
    logger.info('create_masshire_preprofiles started for %s (%s lines)', recruiter_login, len(data))
    recruiter = Staff.objects.get(login=recruiter_login)
    try_create_masshire_preprofiles_by_parsed_data(data, recruiter)
    logger.info('create_masshire_preprofiles finished')


@app.task
def adopt_masshire(person_id: int, masshire_tags: List[str]):
    repository = Repository(Staff.objects.get(id=person_id))
    join_at = datetime.date.today()
    for preprofile in repository.preprofiles_for_masshire_export(masshire_tags):
        ctrl = repository.existing(preprofile=preprofile)
        try:
            ctrl.approve(join_at)
            is_ready = ctrl.try_make_ready()
            if is_ready:
                ctrl.adopt()
            else:
                logger.warning(
                    f'Can\'t adopt {preprofile.id} while masshiring {masshire_tags} due to fail to set uid/guid'
                )
        except Exception as exc:
            logger.warning(f'Can\'t adopt {preprofile.id} while masshiring {masshire_tags} due to {exc}')


@app.task(bind=True, max_retries=3)
def prepare_digital_sign_task(self, code_id, login):
    initiator = Staff.objects.get(login='robot-femida').user
    code = VerifyCode.objects.get(id=code_id)
    prepared, oebs_resp = prepare_digital_sign(code, initiator, login, code.phone_number)
    if not prepared:
        logger.warning(
            f'Cannot prepare digital sign for code {code.id}. '
            f'Attempt {self.request.retries}'
            f' OEBS responded with {oebs_resp}' if oebs_resp else ''
        )
        self.retry(countdown=60)


@app.task(bind=True, max_retries=3)
def create_digital_sign_task(self, object_type: str, object_id: int, phone_id: int):
    phone = StaffPhone.objects.get(id=phone_id)
    code = VerifyCode.objects.filter(
        state=VERIFY_STATE.VERIFIED,
        object_type=object_type,
        object_id=object_id,
    ).first()
    if not code:
        logger.error(
            f'Cannot create digital sign. '
            f'Code not found ({object_type} {object_id})'
        )
        return
    code.phone_number = normalize_phone(code.phone_number)
    phone.number = normalize_phone(phone.number)
    if code.phone_number != phone.number:
        logger.error(
            f'Cannot create digital sign. '
            f'Phones do not match ({code.phone_number} != {phone.number})'
        )
        return

    code.attach_to_staff(phone)

    # Инициатором выпуска ЭЦП в данном случае сейчас всегда является Фемида
    initiator = Staff.objects.get(login='robot-femida').user

    created = create_digital_sign(code, initiator)
    if not created:
        logger.warning(
            f'Cannot create digital sign for code {code.id}. '
            f'Attempt {self.request.retries}'
        )
        self.retry(countdown=60)
