from datetime import date, timedelta
from logging import getLogger
from typing import Any, AnyStr, Dict, List, Optional, Tuple

import blackbox
import dns.resolver
import phonenumbers
import yenv
from django.conf import settings
from django.core.exceptions import ValidationError
from django.db import DatabaseError
from transliterate import translit

from staff.departments.models import Department
from staff.person.models import EMPLOYMENT, Staff
from staff.person_avatar.tasks import upload_preprofile_photo

from staff.preprofile.models.preprofile import (
    CANDIDATE_TYPE,
    CITIZENSHIP_TRANSLATE,
    FORM_TYPE,
    PREPROFILE_STATUS,
    Preprofile,
)


logger = getLogger(__name__)


def launch_person_avatar_task(task_kwargs):
    task = upload_preprofile_photo
    if yenv.type == 'development':
        task(**task_kwargs)
    else:
        task.apply_async(kwargs=task_kwargs, countdown=timedelta(minutes=2).total_seconds())


def get_ip(name):
    try:
        return [item.to_text() for item in dns.resolver.query(name, 'A')]
    except (dns.resolver.NXDOMAIN, dns.resolver.NoAnswer):
        return []


def get_uid_from_passport(login: str) -> Optional[str]:
    answer = blackbox.userinfo(uid_or_login=login, userip='127.0.0.1', by_login=True)
    return answer.get('uid') or None


def user_is_adopter(person):
    """
    :rtype: bool
    :type person: Staff
    """
    return person.user.has_perm('preprofile.add_personadoptapplication')


def _trim_for_vowel(login, gender):  # type: (AnyStr, AnyStr) -> AnyStr
    """
    Если фамилия обрезалась, проверяем, что она обрезалась для М не на гласной букве, для Ж не на согласной.
    Если проверка не прошла, то режем 1 букву из конца, и проверяем ещё раз
    """
    vowel = ['a', 'e', 'i', 'o', 'u', 'y']

    if gender.lower() == 'f':
        while login[-1] not in vowel:
            login = login[:-1]
    else:
        while login[-1] in vowel:
            login = login[:-1]

    return login


def _replace_illegal_symbols(string):  # type: (AnyStr) -> AnyStr
    """Удаляем кавычки, недопустимые в именах и заменяем `j` на `y`"""
    return string.replace('\'', '').replace('j', 'y').replace('J', 'Y')


def generate_masshire_login(
    first: str,
    last: str,
    middle: str,
    gender: Optional[str],
    login: str = None,
) -> Tuple[str, str, str]:
    from staff.preprofile.login_validation import validate_login, MAX_LOGIN_LENGTH

    first_en = _replace_illegal_symbols(translit(first, 'ru', reversed=True))
    last_en = _replace_illegal_symbols(translit(last, 'ru', reversed=True))
    middle_en = _replace_illegal_symbols(translit(middle, 'ru', reversed=True))

    if login:
        return login, first_en, last_en

    templates = [
        '{first:.1}-{middle:.1}-{last}',
        '{first:.2}-{middle:.1}-{last}',
        '{first:.3}-{middle:.1}-{last}',
        '{first:.4}-{middle:.1}-{last}',
        '{first:.5}-{middle:.1}-{last}',
    ]
    for login_template in templates:
        login = login_template.format(
            first=first_en.lower(),
            middle=middle_en.lower(),
            last=last_en.lower(),
        ).replace('--', '-')[:MAX_LOGIN_LENGTH]
        if gender and (last_en.lower() not in login):
            login = _trim_for_vowel(login, gender)
        try:
            validate_login(login, is_robot=False)
            return login, first_en, last_en
        except ValidationError as e:
            logger.info('masshire login %s is not valid. %s Trying next schema.', login, e.code)
    raise ValidationError('Could\'t create login for %s %s', first, last)


def fill_preprofile_model(
    preprofile: Preprofile,
    first: str,
    last: str,
    middle: str,
    first_en: Optional[str] = None,
    last_en: Optional[str] = None,
    login: Optional[str] = None,
    **kwargs
):
    from staff.preprofile.login_validation import validate_login
    if login:
        # don't allow to change wrong login
        try:
            validate_login(login, is_robot=False, preprofile_id=preprofile.id)
        except ValidationError:
            login = preprofile.login
    gen_login, first_en_gen, last_en_gen = generate_masshire_login(first, last, middle, kwargs.get('gender'), login)
    login = login or gen_login
    first_en = first_en or first_en_gen
    last_en = last_en or last_en_gen

    preprofile.form_type = FORM_TYPE.MASS_HIRE
    preprofile.login = login
    preprofile.first_name = first
    preprofile.last_name = last
    preprofile.middle_name = middle
    preprofile.first_name_en = first_en
    preprofile.last_name_en = last_en
    citizenship = kwargs.get('citizenship')
    citizenship = citizenship and citizenship.lower()
    if citizenship in CITIZENSHIP_TRANSLATE:
        preprofile.citizenship = CITIZENSHIP_TRANSLATE[citizenship]
    simple_fields = ['gender', 'masshire_tag', 'email', 'address',
                     'organization_id', 'department_id', 'position', 'phone']
    for field in simple_fields:
        if kwargs.get(field) is not None:
            setattr(preprofile, field, kwargs[field])

    preprofile.join_at = date.today()
    preprofile.office_id = preprofile.office_id or settings.HOMIE_OFFICE_ID
    preprofile.candidate_type = preprofile.candidate_type or CANDIDATE_TYPE.NEW_EMPLOYEE
    preprofile.employment_type = preprofile.employment_type or EMPLOYMENT.FULL
    preprofile.form_type = FORM_TYPE.MASS_HIRE


def try_create_masshire_preprofiles_by_parsed_data(
    parsed_masshire_data: List[Dict[str, Any]],
    recruiter: Staff,
) -> List[Dict[str, Any]]:
    """
    Создаёт препрофайлы, генерируя логины и дозаполняет parsed_masshire_data новыми данными.
    """
    from staff.departments.models import Department
    dep_url_to_id = dict(
        Department.objects
        .values_list('url', 'id')
        .filter(intranet_status=1, url__in={md['department'] for md in parsed_masshire_data})
    )
    ids = [p['id'] for p in parsed_masshire_data if p['id']]
    if ids:
        id_to_existing = {
            it.id: it for it in Preprofile.objects.filter(id__in=ids)
        }
    else:
        id_to_existing = {}
    res = []
    for preprofile_data in parsed_masshire_data:
        res_data = preprofile_data.copy()
        first = preprofile_data.pop('first_name', None)
        last = preprofile_data.pop('last_name', None)
        middle = preprofile_data.pop('middle_name', None)
        dep_url = preprofile_data.pop('department', None)
        if dep_url not in dep_url_to_id:
            logger.warning('Fail to create preprofile for %s %s with wrong dep url %s', first, last, dep_url)
            continue
        try:
            id_ = preprofile_data.get('id')
            preprofile = id_to_existing[id_] if id_ else Preprofile(status=PREPROFILE_STATUS.NEW, recruiter=recruiter)
            if preprofile.id and preprofile.form_type != FORM_TYPE.MASS_HIRE:
                logger.info('Passing edit preprofile %s as masshire due not been masshire', preprofile.id)
                continue
            fill_preprofile_model(
                preprofile=preprofile,
                first=first,
                last=last,
                middle=middle,
                first_en=preprofile_data.pop('first_name_en', None),
                last_en=preprofile_data.pop('last_name_en', None),
                department_id=dep_url_to_id[dep_url],
                **preprofile_data
            )
            preprofile.save()
            logger.info('Masshire preprofile created. login: %s id:%s', preprofile.login, preprofile.id)
        except (DatabaseError, ValidationError) as e:
            logger.warning('Error creating masshire preprofile  %s %s %s. Error:%s', first, last, middle, e)
        else:
            res_data['id'] = preprofile.id
            res_data['login'] = preprofile.login
            res_data['first_name_en'] = preprofile.first_name_en
            res_data['last_name_en'] = preprofile.last_name_en
        res.append(res_data)
    return res


def normalize_phone(number, region='RU'):
    try:
        parsed = phonenumbers.parse(number, region)
    except phonenumbers.NumberParseException as exc:
        logger.warning('Error during parse phone number %s: %s', number, repr(exc))
        return

    return phonenumbers.format_number(
        numobj=parsed,
        num_format=phonenumbers.PhoneNumberFormat.INTERNATIONAL,
    )


def outstaff_or_external_department(department: Department) -> bool:
    return (
        department
        .get_ancestors(include_self=True)
        .filter(id__in=[settings.OUTSTAFF_DEPARTMENT_ID, settings.EXT_DEPARTMENT_ID])
        .exists()
    )
