import json
import logging

from django.contrib.auth.decorators import permission_required
from django.views.decorators.http import require_http_methods

from staff.lib.decorators import (
    responding_json,
    paginated,
    available_for_external,
)
from staff.preprofile import tasks

from staff.rfid.constants import OWNER
from staff.rfid.controllers import Badges

from staff.preprofile.forms.adopt_form import AdoptForm
from staff.preprofile.models import PersonAdoptApplication, Preprofile, STATUS

logger = logging.getLogger(__name__)


@responding_json
@require_http_methods(['GET', 'POST'])
@permission_required('preprofile.add_personadoptapplication')
@available_for_external('preprofile.available_preprofile_for_externals')
def adopt_persons(request):
    adopter = request.user.staff

    if request.method == 'GET':
        form = AdoptForm()
        return form.as_dict()

    try:
        data = json.loads(request.read())
    except ValueError:
        return {'message': 'invalid_json'}, 400

    form = AdoptForm(data=data, base_initial={'adopter': adopter})
    form.is_valid()

    for person in form.cleaned_data['persons']:
        login = person['login']
        rfid_code = person['rfid_code']

        result = (
            Badges()
            .values('preprofile_id', 'first_name', 'last_name')
            .get(rfid__code=rfid_code, owner=OWNER.CANDIDATE)
        )

        preprofile = Preprofile.objects.get(id=result['preprofile_id'])

        logger.info('Putting %s with rfid %s into queue', login, rfid_code)

        PersonAdoptApplication.objects.create(
            login=login,
            rfid_code=rfid_code,
            status=STATUS.NEW,
            preprofile=preprofile,
            first_name=result['first_name'],
            last_name=result['last_name'],
            adopter=adopter,
        )

    if form.is_valid():
        return {}

    return form.errors


@responding_json
@require_http_methods(['GET'])
@paginated(default_limit=10)
@available_for_external('preprofile.available_preprofile_for_externals')
def adopt_status(request, paginator):
    offset = (paginator.page - 1) * paginator.limit
    applications = list(
        PersonAdoptApplication.objects
        .order_by('-id')
        .values(
            'login',
            'status',
            'status_reason',
            'first_name',
            'last_name',
            'preprofile_id',
        )
        [offset:offset + paginator.limit]
    )

    paginator.result = applications
    paginator.total = PersonAdoptApplication.objects.count()
    return paginator


@responding_json
@require_http_methods(['GET', 'POST'])
@available_for_external('preprofile.available_preprofile_for_externals')
@permission_required('preprofile.can_outstaff')
@permission_required('preprofile.add_personadoptapplication')
def adopt_masshire(request, tag):
    tasks.adopt_masshire.delay(person_id=request.user.staff.id, masshire_tags=[tag])
    return {'status': 'ok'}
