import logging
from collections import defaultdict
from datetime import date

from django.db.models import Q
from django.contrib.auth.decorators import permission_required
from django.http import JsonResponse
from django.views.decorators.http import require_GET

from staff.departments.models import DepartmentStaff, DepartmentRoles, Department
from staff.lib import xlsx
from staff.lib.forms import errors as form_errors
from staff.lib.decorators import (
    responding_json,
    available_for_external,
)
from staff.lib.query import get_list_request_param
from staff.person_avatar.controllers import PreprofileAvatarCollection
from staff.person_avatar.models import AvatarMetadata

from staff.preprofile.forms.person_forms_filter_form import PersonFormsFilterForm
from staff.preprofile.models import PREPROFILE_STATUS, INTERNSHIP_FULLTIME
from staff.preprofile.repository import Repository, NoRightsRepositoryError
from staff.preprofile.views.masshire_export import (
    CitizenshipPagePresenter,
    DataSheetPresenter,
    data_model,
)
from staff.preprofile.views.full_export import (
    FullDataSheetPresenter,
    full_data_model,
)
from staff.preprofile.views.views import (
    get_departments_ids_by_hr_partner,
)

logger = logging.getLogger(__name__)


STATUS_MAP = {
    PREPROFILE_STATUS.READY: 40,
    PREPROFILE_STATUS.CLOSED: 50,
}


def map_status_to_number(preprofile):
    preprofile['state'] = STATUS_MAP[preprofile.pop('status')]


@require_GET
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
def certificator_export(request):
    requested_by = request.user.staff

    try:
        preprofiles = list(Repository(requested_by).preprofiles_for_certificator())
    except NoRightsRepositoryError:
        logger.info('User %s is not allowed to see certificator preprofiles', requested_by.login)
        return form_errors.general_error({'message': 'access_denied'}), 403

    for preprofile in preprofiles:
        preprofile['username'] = preprofile.pop('login')
        preprofile['office'] = preprofile.pop('office_id')
        preprofile['department_url'] = preprofile.pop('department__url')

    return preprofiles


@require_GET
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
def puncher_export(request):
    requested_by = request.user.staff

    try:
        preprofiles = list(Repository(requested_by).preprofiles_for_puncher())
    except NoRightsRepositoryError:
        logger.info('User %s is not allowed to see puncher preprofiles', requested_by.login)
        return form_errors.general_error({'message': 'access_denied'}), 403

    for preprofile in preprofiles:
        preprofile['username'] = preprofile.pop('login')
        preprofile['group'] = preprofile['department__group__id']
        map_status_to_number(preprofile)

    return preprofiles


@require_GET
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
def idm_export(request):
    requested_by = request.user.staff

    try:
        preprofiles = list(Repository(requested_by).preprofiles_for_idm())
    except NoRightsRepositoryError:
        logger.info('User %s is not allowed to see idm preprofiles', requested_by.login)
        return form_errors.general_error({'message': 'access_denied'}), 403

    for preprofile in preprofiles:
        preprofile['username'] = preprofile.pop('login')
        map_status_to_number(preprofile)

    return preprofiles


@require_GET
@permission_required('preprofile.can_outstaff')
def masshire_export(request):
    tags = get_list_request_param(request, 'masshire_tag')
    pages = [
        DataSheetPresenter(data_model(request.user.get_profile(), tags)),
        CitizenshipPagePresenter(),
    ]

    file_name = 'masshire_export_on_{}'.format(date.today())
    return xlsx.make_xlsx_file_response(pages, file_name)


@permission_required('preprofile.can_outstaff')
@responding_json
def check_masshire_status(request, tag):
    return dict(
        Repository(request.user.staff)
        .preprofiles_for_masshire_export([tag])
        .values_list('id', 'status')
    )


LIST_FIELDS = [
    ('office', 'office'),
    ('org', 'organization'),
    ('status', 'status'),
]
EXTRA_LIST_FIELDS = [
    ('internship_fulltime', 'internship_fulltime'),
    ('root_department', 'root_department'),
]


@require_GET
@available_for_external('preprofile.available_preprofile_for_externals')
def preprofile_export(request):
    qs = full_data_model(request.user.get_profile())

    filter_form = _get_filter_form(request)

    if not filter_form.is_valid():
        return JsonResponse(filter_form.errors_as_dict(), status=400)

    qs = filter_qs(qs, filter_form)
    filling_qs(list(qs))
    pages = [FullDataSheetPresenter(qs)]

    return xlsx.make_xlsx_file_response(pages, None, file_name_prefix='full_export_on')


def _get_filter_form(request):
    dict_to_form = dict(request.GET.items())
    for form_field, _ in LIST_FIELDS + EXTRA_LIST_FIELDS:
        dict_to_form[form_field] = request.GET.getlist(form_field)

    return PersonFormsFilterForm(data=dict_to_form)


def filter_qs(qs, filter_form):
    for form_field, filter_op in LIST_FIELDS:
        values = filter_form.cleaned_data[form_field]
        if not values:
            values = []
        elif not isinstance(values, list):
            values = [values]
        filter_ = Q()
        for value in values:
            filter_ |= Q((filter_op, value))
        qs = qs.filter(filter_)

    form_field_to_filter_op = [
        ('department', 'department'),
        ('recruiter', 'recruiter'),
        ('date_from', 'join_at__gte'),
        ('date_to', 'join_at__lte'),
    ]

    for form_field, filter_op in form_field_to_filter_op:
        value = filter_form.cleaned_data[form_field]
        if value:
            filter_kwargs = {filter_op: value}
            qs = qs.filter(**filter_kwargs)

    preprofile = filter_form.cleaned_data['preprofile']
    if preprofile:
        qs = qs.filter(id=preprofile.id)

    chief = filter_form.cleaned_data['chief']
    if chief:
        chief_departments = (
            DepartmentStaff
            .objects
            .filter(staff=chief, role_id=DepartmentRoles.CHIEF.value)
            .values_list('department')
        )
        qs = qs.filter(department__in=chief_departments)

    hr_partner = filter_form.cleaned_data['hr_partner']
    if hr_partner:
        hr_partner_departments = get_departments_ids_by_hr_partner(hr_partner)
        qs = qs.filter(department__in=hr_partner_departments)

    root_department = filter_form.cleaned_data['root_department']
    if root_department:
        root_department_tree_ids = Department.objects.filter(id__in=root_department).values_list('tree_id', flat=True)
        qs = qs.filter(department__tree_id__in=root_department_tree_ids)

    internship_fulltime = filter_form.cleaned_data['internship_fulltime']
    if internship_fulltime:
        internship_fulltime_filter = Q()
        for value in internship_fulltime:
            not_internship = value != INTERNSHIP_FULLTIME.INTERNSHIP
            internship_fulltime_filter |= Q(date_completion_internship__isnull=not_internship)
        qs = qs.filter(internship_fulltime_filter)

    return qs


def filling_qs(loaded_data):
    departments = set(x['department__url'] for x in loaded_data)
    department_roles = defaultdict(lambda: defaultdict(list))

    department_roles_qs = (
        DepartmentStaff.objects
        .filter(department__url__in=departments)
        .filter(role_id__in=(DepartmentRoles.CHIEF.value, DepartmentRoles.HR_PARTNER.value))
        .values('department__url', 'role_id', 'staff__login')
    )

    for role_info in department_roles_qs:
        department_roles[role_info['department__url']][role_info['role_id']].append(role_info['staff__login'])

    pre_profile_ids = set(x['id'] for x in loaded_data)
    avatar_metadata_ids_qs = (
        AvatarMetadata.objects
        .filter(preprofile_id__in=pre_profile_ids, is_main=True)
        .values('preprofile_id', 'id')
    )
    metadata_ids = {x['preprofile_id']: x['id'] for x in avatar_metadata_ids_qs}

    for idx, item in enumerate(loaded_data):
        current_roles = department_roles.get(item['department__url'], {})
        chief = current_roles.get(DepartmentRoles.CHIEF.value, None)
        item['chief'] = chief[0] if chief else None
        hr_partners = current_roles.get(DepartmentRoles.HR_PARTNER.value, [])
        item['hr_partners'] = ', '.join(hr_partners)

        avatar_id = metadata_ids.get(item['id'])
        if avatar_id is not None:
            item['photo'] = PreprofileAvatarCollection.get_link_for_id(avatar_id)

        item['idx'] = idx
        item['birthday'] = item['birthday'].isoformat() if item['birthday'] else None
        item['join_at'] = item['join_at'].isoformat() if item['join_at'] else None
        item['modified_at'] = item['modified_at'].isoformat() if item['modified_at'] else None
        item['created_at'] = item['created_at'].isoformat() if item['created_at'] else None
