import logging
import pytz

from django.http import HttpResponseForbidden
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_GET, require_POST, require_http_methods
from staff.lib.models.roles_chain import get_grouped_chiefs_by_departments

from staff.lib.decorators import (
    responding_json,
    consuming_json,
    available_for_external,
    auth_by_tvm_only,
)
from staff.lib.forms import errors as form_errors
from staff.lib.utils.date import parse_datetime
from staff.person_profile.controllers import digital_sign

from staff.preprofile.controllers import ControllerError
from staff.preprofile.controller_behaviours import (
    FemidaEmployeeBehaviour,
    FemidaOutstaffExternalBehaviour,
    FemidaRotationEditBehaviour,
    OutstaffBehaviour,
)
from staff.preprofile.forms import FemidaAttachPhoneDSForm, FemidaVerifyPhoneDSForm
from staff.preprofile.models import (
    CANDIDATE_TYPE,
    FORM_TYPE,
    Preprofile,
)
from staff.preprofile.repository import Repository
from staff.preprofile.views.views import post_new_person_form

logger = logging.getLogger(__name__)


def _format_preprofiles_data(preprofiles):
    fields = [
        'login',
        'join_at',
        'status',
        'hdrfs_ticket',
        'supply_ticket',
        'femida_offer_id',
        'modified_at',
        'candidate_type',
        'organization_id',
        'department_id',
        'modified_at',
        'date_completion_internship',
    ]
    departments = [pp.department for pp in preprofiles]

    chiefs_data = get_grouped_chiefs_by_departments(departments, fields=['login'])
    chiefs = {dep.id: chiefs_data[dep.id].get('staff__login') for dep in departments}

    preprofiles_data = []
    for preprofile in preprofiles:
        preprofile_data = {
            field: getattr(preprofile, field)
            for field in fields
        }
        preprofile_data.update({
            'id': preprofile.id,
            'position': preprofile.position_staff_text,
            'recruiter_login': preprofile.recruiter.login,
            'chief_login': chiefs[preprofile.department_id],
        })
        preprofiles_data.append(preprofile_data)
    return preprofiles_data


@require_GET
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
def femida_export(request):
    if 'date_from' not in request.GET:
        return form_errors.general_error({'message': 'missing__date_from'}), 400

    msk_tz = pytz.timezone('Europe/Moscow')
    date_from = parse_datetime(request.GET['date_from']).astimezone(msk_tz).replace(tzinfo=None)

    preprofiles = (
        Preprofile.objects.filter(femida_offer_id__isnull=False) |
        Preprofile.objects.exclude(ext_form_link='')
    )
    preprofiles = (
        preprofiles.filter(modified_at__gte=date_from)
        .select_related('department', 'recruiter')
        .order_by('id')
    )
    return _format_preprofiles_data(preprofiles)


@require_POST
@responding_json
@consuming_json
@csrf_exempt
@available_for_external('preprofile.available_preprofile_for_externals')
def create_from_femida(request, json_data):
    if request.client_application.name != 'femida':
        logging.info('Application %s is not allowed to create preprofiles', request.client_application.name)
        return HttpResponseForbidden()

    requested_by = request.user.staff
    candidate_type = json_data.get('candidate_type', None)
    form_type = FORM_TYPE.ROTATION if candidate_type == CANDIDATE_TYPE.CURRENT_EMPLOYEE else FORM_TYPE.EMPLOYEE
    return post_new_person_form(json_data, form_type, requested_by)


@require_http_methods(['GET', 'POST'])
@responding_json
@consuming_json
@csrf_exempt
@available_for_external('preprofile.available_preprofile_for_externals')
def femida_update(request, preprofile_id, json_data):
    def create_behaviour(preprofile):
        if preprofile.form_type == FORM_TYPE.EMPLOYEE and preprofile.femida_offer_id:
            return FemidaEmployeeBehaviour()

        if preprofile.form_type == FORM_TYPE.OUTSTAFF:
            return OutstaffBehaviour()

        if preprofile.form_type == FORM_TYPE.ROTATION:
            return FemidaRotationEditBehaviour()

        raise KeyError('Unsupported form_type')

    if not request.client_application or request.client_application.name != 'femida':
        return form_errors.general_error({'message': 'forbidden'}), 403

    try:
        controller = Repository(request.user.staff).existing(preprofile_id, create_behaviour)
    except KeyError:
        return form_errors.general_error({'message': 'invalid_form_type'}), 400
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404

    if request.method == 'GET':
        result = Repository(request.user.staff).preprofile_for_femida(preprofile_id)
        return result, 200

    try:
        controller.try_apply_changes(json_data)
    except ControllerError as e:
        return e.errors_dict, 400

    return {}, 200


@require_POST
@responding_json
@consuming_json
@csrf_exempt
@available_for_external('preprofile.available_preprofile_for_externals')
def femida_submit_external_form(request, preprofile_id, json_data):
    if request.client_application.name != 'femida':
        logging.info('Application %s is not allowed to update preprofiles', request.client_application.name)
        return form_errors.general_error({'message': 'forbidden'}), 403

    try:
        def behaviour_factory(_preprofile):
            return FemidaOutstaffExternalBehaviour()

        controller = Repository(request.user.staff).existing(preprofile_id, behaviour_factory)
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404

    try:
        controller.try_apply_changes(json_data)
        controller.prepare()
    except ControllerError as e:
        return e.errors_dict, 400

    return {}, 200


@require_POST
@responding_json
@consuming_json
@csrf_exempt
@available_for_external('preprofile.available_preprofile_for_externals')
@auth_by_tvm_only(['femida'])
def femida_ds_attach_phone(request, json_data):
    form = FemidaAttachPhoneDSForm(data=json_data)
    if not form.is_valid():
        return form.errors_as_dict(), 400

    result = digital_sign.attach_phone(initiator=request.user, **form.cleaned_data)
    if not result.ok:
        return form_errors.general_error({'message': result.status}), 400
    return {}


@require_POST
@responding_json
@consuming_json
@csrf_exempt
@available_for_external('preprofile.available_preprofile_for_externals')
@auth_by_tvm_only(['femida'])
def femida_ds_verify_phone(request, json_data):
    form = FemidaVerifyPhoneDSForm(data=json_data)
    if not form.is_valid():
        return form.errors_as_dict(), 400

    status, code = digital_sign.verify_code(initiator=request.user, **form.cleaned_data)
    if status != digital_sign.VERIFY_ANSWERS.OK:
        return form_errors.general_error({'message': status}), 400
    return {}
