import logging

from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_GET, require_POST
from django_yauth.authentication_mechanisms.tvm import TvmServiceRequest

from staff.lib.forms import errors as form_errors
from staff.lib.decorators import (
    available_by_tvm,
    available_for_external,
    consuming_json,
    responding_json,
)

from staff.preprofile.controller_behaviours import HelpdeskBehaviour
from staff.preprofile.controllers import ControllerError
from staff.preprofile.models import Preprofile
from staff.preprofile.repository import Repository, NoRightsRepositoryError

logger = logging.getLogger(__name__)


@require_GET
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
@available_by_tvm(['taxi_effrat-employees'])
def helpdesk_export(request):
    yauser = request.yauser
    if not isinstance(yauser, TvmServiceRequest):
        if not Repository.user_can_view_all_preprofiles(request.user):
            return form_errors.general_error({'message': 'access_denied'}), 403

    preprofiles = list(Repository.preprofiles_for_helpdesk())

    return preprofiles


@require_POST
@responding_json
@consuming_json
@csrf_exempt
@available_for_external('preprofile.available_preprofile_for_externals')
def update_helpdesk_ticket(request, preprofile_id, json_data):
    try:
        controller = (
            Repository(request.user.staff)
            .existing(preprofile_id, lambda preprofile: HelpdeskBehaviour())
        )
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404
    except NoRightsRepositoryError:
        return form_errors.general_error({'message': 'forbidden'}), 403

    try:
        controller.try_apply_changes(json_data)
    except ControllerError as e:
        return e.errors_dict, 400

    return {'message': 'ok'}, 200
