import logging
from typing import Any, Dict, List, Iterator

import attr
import xlrd

from django.contrib.auth.decorators import permission_required
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST

from staff.lib.decorators import responding_json, available_for_external
from staff.lib.forms import errors as form_errors

from staff.preprofile.forms import FileImportForm
from staff.preprofile.tasks import create_masshire_preprofiles

logger = logging.getLogger(__name__)


def preprofiles_data(import_file):
    book = xlrd.open_workbook(file_contents=import_file.read())
    sheet = book.sheet_by_index(0)
    return preprofile_tuples_to_dict(sheet.get_rows())


@attr.s(auto_attribs=True)
class PreprofileFileCell:
    value: Any


def preprofile_tuples_to_dict(rows: Iterator[List[PreprofileFileCell]]) -> List[Dict[str, Any]]:
    file_to_dict_name = {
        'тег массового найма': 'masshire_tag',
        'анкета в наниматоре': 'id',
        'фамилия': 'last_name',
        'имя': 'first_name',
        'отчество': 'middle_name',
        'first name': 'first_name_en',
        'last name': 'last_name_en',
        'login': 'login',
        'пол': 'gender',
        'e-mail': 'email',
        'телефон': 'phone',
        'адрес': 'address',
        'id юридического лица': 'organization_id',
        'подразделение': 'department',
        'должность': 'position',
        'гражданство': 'citizenship',
    }

    headers = [it.value.lower().strip('* ') for it in next(rows)]

    def cells_val(cell):
        return '{:.0f}'.format(cell.value) if isinstance(cell.value, float) else cell.value

    res = []
    for row in rows:
        parsed = {}
        for i, val in enumerate(row):
            field = file_to_dict_name.get(headers[i])
            if field is None:
                continue
            val = cells_val(val)
            if field == 'department':
                val = val.split('/')[-1]
            elif field == 'id':
                val = val and int(val)
            parsed[field] = val
        res.append(parsed)

    return res


@require_POST
@csrf_exempt
@responding_json
@permission_required('preprofile.can_outstaff')
@available_for_external('preprofile.available_preprofile_for_externals')
def outstaff_mass_import(request):
    form = FileImportForm(request.POST, request.FILES)
    if not form.is_valid():
        return form_errors.sform_general_error('missing_file'), 400

    try:
        parsed_preprofiles_data = preprofiles_data(request.FILES['import_file'])
    except (xlrd.XLRDError, IndexError):
        return form_errors.sform_general_error('incorrect_template'), 400

    create_masshire_preprofiles.delay(recruiter_login=request.user.staff.login, data=parsed_preprofiles_data)

    return {'message': 'ok'}, 200
