from typing import Iterable, Optional

from staff.lib import xlsx
from staff.preprofile.models import CITIZENSHIP
from staff.preprofile.views.masshire_export import CitizenshipPagePresenter


_HEADINGS = [
    'N/N',
    'Тег массового найма*',
    'Анкета в наниматоре',
    'Фамилия*',
    'Имя*',
    'Отчество*',
    'First Name*',
    'Last Name*',
    'Login',
    'Пол*',
    'E-mail',
    'Телефон',
    'Адрес',
    'ID Юридического лица*',
    'Подразделение*',
    'Должность',
    'Гражданство*',
]

CITIZENSHIP_TRANSLATE = {
    CITIZENSHIP.RUSSIAN: 'Российское',
    CITIZENSHIP.BELORUSSIAN: 'Белорусское',
    CITIZENSHIP.UKRAINIAN: 'Украинское',
    CITIZENSHIP.KAZAKHSTAN: 'Казахстанское',
    CITIZENSHIP.KYRGYZSTAN: 'Киргизское',
    CITIZENSHIP.ARMENIA: 'Армянское',
    CITIZENSHIP.OTHER: 'Другое',
}


class DataSheetPresenter(xlsx.SheetPresenter):
    sheet_name = 'Данные'

    _font_base = {
        'font_name': 'Yandex Sans Display Light',
        'font_size': 12,
    }

    _heading_row_format = {
        'align': 'center',
        'valign': 'vcenter',
        'text_wrap': True,
        'bold': True,
        'bg_color': '#B4C6E7',
        **_font_base,
    }

    _data_row_format = {
        'border': 1,
        **_font_base,
    }

    _url_cell_format = {
        'border': 1,
        'underline': True,
        'font_color': '#0000FF',
        'font_size': 12,
    }

    data_validations = (
        xlsx.DataValidation('Q2:Q1048576', 'list', f'={CitizenshipPagePresenter.sheet_name}!$A$2:$A$8'),
    )

    def __init__(self, model: Iterable):
        super().__init__(model=model)

    def columns(self) -> Iterable[xlsx.Column]:
        return [
            xlsx.Column(width=10),  # NN
            xlsx.Column(width=10),  # tag
            xlsx.Column(width=19.25),  # preprofile id
            xlsx.Column(width=25.42),  # last name
            xlsx.Column(width=20.58),  # first name
            xlsx.Column(width=25.42),  # middle name
            xlsx.Column(width=16.25),  # first name en
            xlsx.Column(width=14.08),  # last name en
            xlsx.Column(width=14.08),  # last name en
            xlsx.Column(width=18.83),  # login
            xlsx.Column(width=10),  # gender
            xlsx.Column(width=28.08),  # email
            xlsx.Column(width=25.08),  # phone
            xlsx.Column(width=19.75),  # address
            xlsx.Column(width=22.75),  # department
            xlsx.Column(width=32.92),  # position
            xlsx.Column(width=17.58),  # citizenship
        ]

    def rows(self) -> Iterable[xlsx.ViewRow]:
        yield self.column_captions_row()
        yield from self.data_rows()

    def data_rows(self):
        for idx, row in enumerate(self._model):
            department_url = row['department__url']
            department = f'https://staff.yandex-team.ru/departments/{department_url}'
            cells = [
                xlsx.ViewCell(text=str(idx+1)),
                xlsx.ViewCell(row['masshire_tag']),
                xlsx.ViewCell(row['id']),
                xlsx.ViewCell(row['first_name']),
                xlsx.ViewCell(row['last_name']),
                xlsx.ViewCell(row['middle_name']),
                xlsx.ViewCell(row['first_name_en']),
                xlsx.ViewCell(row['last_name_en']),
                xlsx.ViewCell(row['login']),
                xlsx.ViewCell(row['gender']),
                xlsx.ViewCell(row['email'], url=row['email'], cell_format=self._url_cell_format),
                xlsx.ViewCell(row['phone']),
                xlsx.ViewCell(row['address']),
                xlsx.ViewCell(row['organization__name']),
                xlsx.ViewCell(department, url=department, cell_format=self._url_cell_format),
                xlsx.ViewCell(row['position']),
                xlsx.ViewCell(self.citizenship_text(row['citizenship'])),
            ]

            yield xlsx.ViewRow(cells=cells, row_format=self._data_row_format)

    def citizenship_text(self, citizenship) -> Optional[str]:
        return CITIZENSHIP_TRANSLATE.get(citizenship)

    def column_captions_row(self) -> xlsx.ViewRow:
        return xlsx.ViewRow(
            cells=xlsx.SheetPresenter.default_view_cells_from_row_data(_HEADINGS),
            row_format=self._heading_row_format,
        )
