import logging
from typing import List, Dict
from datetime import date
from django.conf import settings
from django.contrib.auth.decorators import permission_required
from django.core.urlresolvers import reverse
from django.http import HttpResponseForbidden, HttpResponse, Http404, JsonResponse
from django.template import loader, Context
from django.views.decorators.csrf import csrf_exempt, ensure_csrf_cookie
from django.views.decorators.http import require_http_methods, require_GET, require_POST
from django.db.models import Q

from staff.person.models import Staff
from staff.departments.models import DepartmentStaff, DepartmentRoles, Department, ProposalMetadata
from staff.departments.edit.proposal_mongo import to_mongo_id, MONGO_COLLECTION_NAME, from_mongo_id


from staff.lib.decorators import (
    available_for_external,
    responding_json,
    paginated,
    consuming_json,
)
from staff.lib.mongodb import mongo
from staff.lib.forms import errors as form_errors
from staff.lib.utils.qs_values import extract_related
from staff.lib.models.roles_chain import get_grouped_hrbp_by_departments

from staff.preprofile.controllers import ControllerError, AccessDeniedControllerError
from staff.preprofile.controller_behaviours import FemidaEmployeeBehaviour
from staff.preprofile import notifications
from staff.preprofile.forms import CheckLoginForm, PersonFormsFilterForm, RotationHistoryForm
from staff.preprofile.models_and_forms_utils import fill_chief_and_departments_data
from staff.preprofile.models import Preprofile, FORM_TYPE, PREPROFILE_STATUS, INTERNSHIP_FULLTIME
from staff.preprofile.repository import is_valid_form_type, Repository, NoRightsRepositoryError

logger = logging.getLogger(__name__)


@require_GET
@responding_json
@paginated
@available_for_external('preprofile.available_preprofile_for_externals')
def person_forms(request, paginator):
    left_bound = (paginator.page - 1) * paginator.limit
    right_bound = left_bound + paginator.limit

    fields = [
        'id',
        'first_name',
        'middle_name',
        'last_name',
        'status',
        'created_at',
        'modified_at',
        'login',
        'organization__name',
        'department__name',
        'department__id',
        'department__tree_id',
        'department__lft',
        'department__rght',
        'department__url',
        'position',
        'office__name',
        'office__city__name',
        'join_at',
    ]

    qs = (
        Repository(request.user.staff)
        .preprofiles_qs()
        .select_related('department', 'office', 'organization', 'office__city')
        .values(*fields)
        .order_by('-join_at')
    )

    list_fields = [
        ('office', 'office'),
        ('org', 'organization'),
        ('status', 'status'),
    ]

    extra_list_fields = [
        ('internship_fulltime', 'internship_fulltime'),
        ('root_department', 'root_department'),
    ]

    dict_to_form = dict(request.GET.items())
    for form_field, _ in list_fields + extra_list_fields:
        dict_to_form[form_field] = request.GET.getlist(form_field)

    filter_form = PersonFormsFilterForm(data=dict_to_form)

    if not filter_form.is_valid():
        return filter_form.errors_as_dict(), 400

    form_field_to_filter_op = [
        ('department', 'department'),
        ('recruiter', 'recruiter'),
        ('date_from', 'join_at__gte'),
        ('date_to', 'join_at__lte'),
    ]

    for form_field, filter_op in list_fields:
        values = filter_form.cleaned_data[form_field]
        if not values:
            values = []
        elif not isinstance(values, list):
            values = [values]

        filter_ = Q()
        for value in values:
            filter_ |= Q((filter_op, value))
        qs = qs.filter(filter_)

    for form_field, filter_op in form_field_to_filter_op:
        value = filter_form.cleaned_data[form_field]
        if value:
            filter_kwargs = {filter_op: value}
            qs = qs.filter(**filter_kwargs)

    preprofile = filter_form.cleaned_data['preprofile']
    if preprofile:
        qs = qs.filter(id=preprofile.id)

    chief = filter_form.cleaned_data['chief']
    if chief:
        chief_departments = (
            DepartmentStaff.objects
            .filter(staff=chief, role_id=DepartmentRoles.CHIEF.value)
            .values_list('department')
        )
        qs = qs.filter(department__in=chief_departments)

    hr_partner = filter_form.cleaned_data['hr_partner']
    if hr_partner:
        hr_partner_departments = get_departments_ids_by_hr_partner(hr_partner)
        qs = qs.filter(department__in=hr_partner_departments)

    root_department = filter_form.cleaned_data['root_department']
    if root_department:
        root_department_tree_ids = Department.objects.filter(id__in=root_department).values_list('tree_id', flat=True)
        qs = qs.filter(department__tree_id__in=root_department_tree_ids)

    internship_fulltime = filter_form.cleaned_data['internship_fulltime']
    if internship_fulltime:
        internship_fulltime_filter = Q()
        for value in internship_fulltime:
            not_internship = value != INTERNSHIP_FULLTIME.INTERNSHIP
            internship_fulltime_filter |= Q(date_completion_internship__isnull=not_internship)
        qs = qs.filter(internship_fulltime_filter)

    total = qs.count()

    if left_bound >= total:
        if paginator.page == 1:
            paginator.result = []
            paginator.total = 0
            return paginator

        return form_errors.general_error({'message': 'not_found'}), 404

    preprofiles = qs[left_bound:right_bound]

    fill_chief_and_departments_data(preprofiles)

    for preprofile in preprofiles:
        preprofile.update(extract_related(preprofile, 'office__city', 'city'))
        preprofile.update(extract_related(preprofile, 'office', 'office'))
        preprofile.update(extract_related(preprofile, 'organization', 'organization'))

    paginator.result = list(preprofiles)
    paginator.total = total
    return paginator


@require_http_methods(['GET', 'POST'])
@ensure_csrf_cookie
@responding_json
@consuming_json
@available_for_external('preprofile.available_preprofile_for_externals')
def new_form(request, form_type, json_data):
    requested_by = request.user.staff

    if form_type in [FORM_TYPE.EMPLOYEE, FORM_TYPE.MASS_HIRE]:
        return form_errors.general_error({'message': 'invalid_type'}), 400

    if not is_valid_form_type(form_type):
        return form_errors.general_error({'message': 'invalid_type'}), 400

    if request.method == 'POST':
        return post_new_person_form(json_data, form_type, requested_by)

    try:
        result = Repository(requested_by).new(form_type).front_data()
        return result
    except NoRightsRepositoryError:
        return form_errors.general_error({'message': 'forbidden'}), 403


@require_POST
@csrf_exempt
@consuming_json
@available_for_external('preprofile.available_preprofile_new_form_api_for_externals')
def new_form_api(request, form_type, json_data):
    requested_by = request.user.staff
    permission_by_form_type = {
        FORM_TYPE.OUTSTAFF: 'preprofile.can_create_outstaff_by_api',
        FORM_TYPE.EXTERNAL: 'preprofile.can_create_external_by_api',
        FORM_TYPE.ROBOT: 'preprofile.can_create_robot_by_api',
        FORM_TYPE.ZOMBIE: 'preprofile.can_create_zombie_by_api',
    }

    if form_type not in permission_by_form_type:
        return JsonResponse(form_errors.general_error({'message': 'invalid_type'}), status=400)

    if not requested_by.user.has_perm(permission_by_form_type[form_type]):
        logger.info(
            'User %s is not allowed to create preprofiles of type %s by api',
            requested_by.login,
            form_type,
        )
        return JsonResponse(form_errors.general_error({'message': 'access_denied'}), status=403)
    result, status = post_new_person_form(json_data, form_type, requested_by)
    return JsonResponse(result, status=status)


def post_new_person_form(data, form_type, requested_by):
    try:
        controller = Repository(requested_by).new(form_type)
    except NoRightsRepositoryError:
        logging.info('User %s is not allowed to create preprofiles', requested_by.login)
        return form_errors.general_error({'message': 'forbidden'}), 403

    try:
        controller.try_apply_changes(data)
    except ControllerError as e:
        errors = e.errors_dict

        if 'femida_offer_id' not in errors['errors']:
            return errors, 400

        femida_offer_id = errors['errors']['femida_offer_id'][0]['params']
        preprofile_id = Repository(requested_by).preprofile_id_by_femida_offer_id(femida_offer_id)
        return {'id': preprofile_id}, 409

    return {'id': controller.preprofile_id}, 200


@require_POST
@csrf_exempt
@available_for_external('preprofile.available_preprofile_for_externals')
def cancel_form(request, preprofile_id):
    if request.client_application.name != 'femida':
        return HttpResponseForbidden()

    try:
        def behaviour_factory(preprofile):
            return FemidaEmployeeBehaviour()
        controller = Repository(request.user.staff).existing(preprofile_id, behaviour_factory)
    except Preprofile.DoesNotExist:
        raise Http404('Preprofile not found')

    controller.cancel()
    return HttpResponse()


@require_http_methods(['GET', 'POST'])
@ensure_csrf_cookie
@responding_json
@consuming_json
@available_for_external('preprofile.available_preprofile_for_externals')
def edit_form(request, preprofile_id, json_data):
    requested_by = request.user.staff

    try:
        controller = Repository(requested_by).existing(preprofile_id)
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404
    except NoRightsRepositoryError:
        return form_errors.general_error({'message': 'forbidden'}), 403

    if request.method == 'GET':
        result = controller.front_data()
        return result

    try:
        controller.try_apply_changes(json_data)
    except ControllerError as e:
        return e.errors_dict, 400

    return {'id': preprofile_id}, 200


@require_POST
@responding_json
@csrf_exempt
@consuming_json
@available_for_external('preprofile.available_preprofile_for_externals')
def check_login_availability(request, json_data):
    form = CheckLoginForm(json_data)

    if not form.is_valid():
        return form.errors_as_dict(), 400

    return {}, 200


@require_POST
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
def approve(request, preprofile_id):
    try:
        controller = Repository(request.user.staff).existing(preprofile_id)
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404
    except NoRightsRepositoryError:
        return form_errors.general_error({'message': 'forbidden'}), 403

    try:
        controller.approve()
    except ControllerError as e:
        return e.errors_dict, 400

    return {'message': 'ok'}, 200


@require_POST
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
def adopt(request, preprofile_id):
    try:
        controller = Repository(request.user.staff).existing(preprofile_id)
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404
    except NoRightsRepositoryError:
        return form_errors.general_error({'message': 'forbidden'}), 403

    try:
        controller.adopt()
    except ControllerError as e:
        return e.errors_dict, 400
    except AccessDeniedControllerError:
        return form_errors.general_error({'message': 'access_denied'}), 403

    return {'message': 'ok'}, 200


@require_GET
@available_for_external('preprofile.available_preprofile_for_externals')
def render_notification(request, id, type_):
    try:
        model = Preprofile.objects.get(id=id)
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404
    try:
        staff_model = Staff.objects.get(login=model.login)
    except Staff.DoesNotExist:
        staff_model = None
    notification_cls = notifications.NOTIFICATION_TYPES.get(type_)
    if notification_cls is None:
        err = 'type_ have to be one of: {}'.format(','.join(notifications.NOTIFICATION_TYPES))
        return form_errors.general_error({'message': err}), 404
    tmpl = loader.get_template(notification_cls.get_template())
    page = str(tmpl.render(Context(dict(
        preprofile=model,
        staff=staff_model,
        **request.GET.dict()
    ))))
    page = page.replace('\r\n', '\n').replace('\n', '<br>\n')
    return HttpResponse(page)


@require_GET
@available_for_external('preprofile.available_preprofile_for_externals')
def notification_list(request, id):
    response_str = '\n'.join(
        '<a href="{}">{}</a><br/>'.format(
            reverse('preprofile:render_notification', kwargs={
                'id': id,
                'type_': template_name,
            }),
            template_name,
        )
        for template_name in sorted(notifications.NOTIFICATION_TYPES)
    )
    return HttpResponse(response_str)


@require_POST
@responding_json
@available_for_external('preprofile.available_preprofile_for_externals')
def create_link(request, preprofile_id):
    try:
        controller = Repository(request.user.staff).existing(preprofile_id)
    except Preprofile.DoesNotExist:
        return form_errors.general_error({'message': 'not_found'}), 404

    try:
        ext_link = controller.create_link()
    except ControllerError as e:
        return e.errors_dict, 400

    return {'link': ext_link}, 200


@require_GET
@responding_json
@permission_required('preprofile.can_see_rotation_history')
def person_rotation_history(request):
    form = RotationHistoryForm(request.GET)
    if not form.is_valid():
        return form.errors_as_dict(), 400

    from_date = form.cleaned_data['from_date']
    to_date = form.cleaned_data['to_date']
    person = form.cleaned_data['person']

    movements_with_proposal = get_proposals_rotations(person.login, from_date, to_date)
    movements_with_femida = get_femida_rotations(person.login, from_date, to_date)

    return {
        'proposals': movements_with_proposal,
        'rotations': movements_with_femida,
    }


def get_proposals_rotations(login: str, from_date: date, to_date: date) -> List[Dict[str, str]]:
    applied_proposals = dict(
        ProposalMetadata.objects
        .filter(applied_at__gte=from_date.isoformat(), applied_at__lte=to_date.isoformat())
        .values_list('proposal_id', 'applied_at')
    )
    mongo_query = {
        '_id': {'$in': [to_mongo_id(_id) for _id in applied_proposals]},
        'persons.actions': {
            '$elemMatch': {
                'login': login,
                'department.changing_duties': True,
            }
        }
    }

    collection = mongo.db.get_collection(MONGO_COLLECTION_NAME)
    proposals_personal_actions = {
        from_mongo_id(p['_id']): p['persons']['actions']
        for p in collection.find(mongo_query, {'persons.actions': 1})
    }

    result = []
    for proposal_id, applied_at in applied_proposals.items():
        personal_actions = proposals_personal_actions.get(proposal_id, [])
        for action in personal_actions:
            if action['login'] == login:
                result.append({
                    'applied_at': applied_at,
                    'new_department': action['department'].get('department', 'NEW')
                })
                break

    return sorted(result, key=lambda x: x['applied_at'])


def get_femida_rotations(login: str, from_date: date, to_date: date) -> List[Dict[str, str]]:
    preprofile_data = (
        Preprofile.objects
        .filter(
            login=login,
            form_type=FORM_TYPE.ROTATION,
            status=PREPROFILE_STATUS.CLOSED,
            join_at__gte=from_date,
            join_at__lte=to_date,
        )
        .values_list('join_at', 'department__url')
        .order_by('join_at')
    )
    return [
        {
            'applied_at': join_at.isoformat(),
            'new_department': dep_url,
        }
        for join_at, dep_url in preprofile_data
    ]


def user_have_perimssion_for_attachments(user):
    if user.is_superuser:
        return True

    person = user.get_profile()
    permitted_department_urls = settings.CAN_ACCESS_ATTACHMENTS_DEPARTMENTS
    permitted_departments = Department.objects.filter(
        url__in=permitted_department_urls,
        tree_id=person.department.tree_id
    )

    for permitted_department in permitted_departments:
        if person.department.is_descendant_of(permitted_department):
            return True

    return False


def _get_all_departments_by_department_staffs(department_staffs, include_self=True):
    departments_query = Department.objects.none()
    for department_staff in department_staffs:
        departments_query |= department_staff.department.get_descendants(include_self=include_self)
    return departments_query.filter(intranet_status=1)


def get_departments_ids_by_hr_partner(hr_partner):
    department_staffs = (
        DepartmentStaff.objects
        .filter(staff=hr_partner, role_id=DepartmentRoles.HR_PARTNER.value)
        .select_related('department')
    )

    departments = _get_all_departments_by_department_staffs(department_staffs)
    departments_with_hr_partners = get_grouped_hrbp_by_departments(departments)

    return [
        department
        for department, hr_partners in departments_with_hr_partners.items()
        if {'id': hr_partner.id} in hr_partners
    ]
