from django.contrib import admin

from staff.lib.forms.fields import DepartmentField, StaffField
from staff.lib.utils.admin import StaffModelAdmin
from staff.proposal.models import CityAttrs, DepartmentAttrs, Grade, ProposalTasksQueue


class _StaffField(StaffField):
    def has_changed(self, initial, data):
        return super(_StaffField, self).has_changed(initial, data[1])


class CityAttrsAdmin(StaffModelAdmin):
    staff_field = 'hr'
    list_display = 'city', 'hr', 'component'


class DepartmentAttrsM2MStaffInline(admin.TabularInline):
    model = None

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == 'staff':
            return _StaffField()
        return super(DepartmentAttrsM2MStaffInline, self).formfield_for_foreignkey(
            db_field, request, **kwargs
        )

    def has_add_permission(self, request):
        return request.user.has_perm('wish_form.add_departmentattrs')

    def has_change_permission(self, request, obj=None):
        return request.user.has_perm('wish_form.change_departmentattrs')

    def has_delete_permission(self, request, obj=None):
        return request.user.has_perm('wish_form.delete_departmentattrs')


class BudgetNotifyInline(DepartmentAttrsM2MStaffInline):
    model = DepartmentAttrs.budget_notify.through
    extra = 2


class TicketAccessInline(DepartmentAttrsM2MStaffInline):
    model = DepartmentAttrs.ticket_access.through
    extra = 2


class DepartmentAttrsAdmin(StaffModelAdmin):
    inlines = [BudgetNotifyInline, TicketAccessInline]
    exclude = 'budget_notify', 'ticket_access'
    list_display = (
        'department',
        'analyst',
        'budget_owner',
        'budget_tag',
        'budget_notify_list',
        'ticket_access_list',
    )

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name in ['analyst', 'budget_owner']:
            return _StaffField(required=False, label=db_field.verbose_name)

        if db_field.name == 'department':
            return DepartmentField(required=False, label=db_field.verbose_name)

        return super(DepartmentAttrsAdmin, self).formfield_for_foreignkey(
            db_field, request, **kwargs)

    def budget_notify_list(self, obj):
        return ', '.join(
            '{0.first_name} {0.last_name}'.format(person)
            for person in obj.budget_notify.all()
        )
    budget_notify_list.short_description = 'список наблюдателей'

    def ticket_access_list(self, obj):
        return ', '.join(
            '{0.first_name} {0.last_name}'.format(person)
            for person in obj.ticket_access.all()
        )
    ticket_access_list.short_description = 'список имеющих доступ'


class GradeAdmin(admin.ModelAdmin):
    list_display = 'sort_order', 'name', 'name_en', 'level'
    ordering = 'sort_order',


class ProposalTasksQueueAdmin(StaffModelAdmin):
    list_display = ('id', 'module', 'callable', 'fail_count', 'args')
    ordering = ('id',)


admin.site.register(CityAttrs, CityAttrsAdmin)
admin.site.register(DepartmentAttrs, DepartmentAttrsAdmin)
admin.site.register(Grade, GradeAdmin)
admin.site.register(ProposalTasksQueue, ProposalTasksQueueAdmin)
