import hashlib
import json
import logging
import time
from typing import Optional

from django.conf import settings

from staff.lib import requests
from staff.departments.controllers.proposal import ApprovementTicketType


logger = logging.getLogger('staff.proposal.controllers.approvement')

ok_session = requests.Session()
ok_session.headers.update({
    'Authorization': f'OAuth {settings.ROBOT_STAFF_OK_OAUTH_TOKEN}',
    'Content-Type': 'application/json',
})


def stop(ticket_key: str, proposal_ctx) -> bool:
    approvement_uuid = proposal_ctx.proposal.get_approvement_uuid(ticket_key)
    if not approvement_uuid:
        logger.warning(f'No approvement uuid to stop approvement')
        return True
    response = ok_session.post(
        f'https://{settings.OK_HOST}/api/approvements/{approvement_uuid}/close/',
        timeout=settings.OK_REQUEST_TIMEOUT,
    )
    if response.status_code not in (200, 403):  # for already closed will be returned 403
        logger.warning(f'Failed to stop approvement. Code {response.status_code} reason: {response.content}')
        return False
    return True


def create(
    proposal_ctx,
    ticket_key: str,
    ticket_type: str,
    staff_uid: Optional[str],
) -> Optional[str]:
    flow_context = proposal_ctx.proposal.get_all_actions().as_dict()
    flow_context['ticket_type'] = ticket_type
    flow_context['ticket_person'] = None
    if ticket_type == ApprovementTicketType.PERSONAL.value:
        login_to_ticket = proposal_ctx.proposal_object['tickets']['persons']
        ticket_person = next(
            (
                login for login, key in login_to_ticket.items()
                if key == ticket_key
            ),
            None,
        )
        flow_context['ticket_person'] = ticket_person
    # empty uid is old behaviour, but it have to be supported coz of old data
    staff_uid = staff_uid or str(time.time())

    logger.info('Calling ok service for ticket %s with uid %s', ticket_key, staff_uid)
    response = ok_session.post(
        f'https://{settings.OK_HOST}/api/approvements/',
        data=json.dumps({
            'flow_context': flow_context,
            'flow_name': 'staff_proposal',
            'object_id': ticket_key,
            'is_parallel': False,
            'text': 'Создано автоматически',
            'uid': staff_uid,
            'groups': settings.ANALYTICS_GROUPS_TO_EDIT_OK,
        }),
        timeout=settings.OK_REQUEST_TIMEOUT,
    )

    try:
        resp_json = response.json()
    except json.JSONDecodeError:
        raise ValueError(f'Wrong format of {response.status_code} response:\n{response.content}')

    if response.status_code == 400:
        try:
            uid, uuid = next(
                (it['params']['uid'], it['params']['uuid'])
                for it in resp_json['error']
                if it['code'] == 'already_exists'
            )
            # uid != staff_uuid means that someone created approvement manually
            return uuid if staff_uid and uid == str_to_md5(staff_uid) else None
        except (KeyError, StopIteration):
            pass
    if (
        resp_json.get('detail', {}).get('error') == 'Flow dropped'
        or resp_json.get('error', [{}])[0].get('code') == 'No stages for flow'
    ):
        return None

    if resp_json.get('uuid'):
        return resp_json['uuid']

    raise ValueError(f'Wrong format of response:\n{resp_json}')


def str_to_md5(string: str) -> str:
    string_utf8 = string.encode()
    return hashlib.md5(string_utf8).hexdigest()
