from typing import Iterable, Any, List, Dict, Optional
import attr

import logging

from staff.person.models import Staff
from staff.lib import waffle
from staff.lib.models.mptt import get_ancestors_query
from staff.lib.startrek.issues import get_issues

from staff.departments.models import Department
from staff.departments.controllers.tickets import ProposalTicketDispatcher
from staff.departments.controllers.proposal import ProposalCtl
from staff.departments.controllers.actions.department_action import DepartmentAction
from staff.departments.controllers.actions.person_action import PersonAction


logger = logging.getLogger(__name__)


MAX_TICKETS_PER_ST_REQUEST = 50


@attr.s(auto_attribs=True)
class ProposalSplittingData:
    persons: List[str] = attr.Factory(list)  # login
    departments: List[int] = attr.Factory(list)  # id


class ProposalSplitting:
    approval_field = 'approvementStatus'
    approval_field_value = 'Согласовано'

    def __init__(self, proposal_ctl: ProposalCtl):
        self._proposal_ctl = proposal_ctl
        self._ticket_dispatcher = ProposalTicketDispatcher.from_proposal_id(proposal_ctl.proposal_id)
        self._splitting_data = None

        self._person_action_by_login = {
            action_dict['login']: PersonAction.from_dict(action_dict)
            for action_dict in proposal_ctl.person_actions
        }

        self._department_actions = [
            DepartmentAction.from_dict(action_dict)
            for action_dict in proposal_ctl.department_actions
        ]

    def _has_move_to_department_being_created(self) -> bool:
        return any(
            action.has_dep_change() and action.department.fake_department
            for action in self._person_action_by_login.values()
        )

    def _department_being_deleted_with_unapproved_persons(self, unapproved_persons: List[str]) -> List[Optional[int]]:
        unapproved_persons_departments = (
            Department.objects
            .filter(id__in=(
                Staff.objects
                .filter(login__in=unapproved_persons)
                .values_list('department_id')
            ))
        )

        departments_that_forbidden_to_delete_query = get_ancestors_query(
            unapproved_persons_departments,
            include_self=True,
        )

        if not departments_that_forbidden_to_delete_query:
            return []

        departments_that_forbidden_to_delete = set(
            Department.objects
            .filter(departments_that_forbidden_to_delete_query)
            .values_list('id', flat=True)
        )

        return [
            department_action.id
            for department_action in self._department_actions
            if department_action.delete and department_action.id in departments_that_forbidden_to_delete
        ]

    def _all_personal_tickets_created(self):
        persons_with_personal_ticket = {action['login'] for action in self._ticket_dispatcher.personal}
        return set(self._proposal_persons_tickets().values()) == persons_with_personal_ticket

    def _has_vacancies_actions(self):
        return self._proposal_ctl.proposal_object['vacancies']['actions']

    def can_split(self) -> bool:
        if not waffle.switch_is_active('enable_proposal_splitting'):
            return False

        if not self._all_personal_tickets_created():
            return False

        if len(list(self._proposal_ctl.vacancy_actions)) > 0 or len(list(self._proposal_ctl.headcount_actions)) > 0:
            return False

        try:
            splitting_data = self.splitting_data()
        except Exception:
            logger.exception('Cannot get splitting data for proposal %s', self._proposal_ctl.proposal_id)
            return True

        if not splitting_data.persons and not splitting_data.departments:
            return False

        if len(splitting_data.persons + splitting_data.departments) == len(list(self._proposal_ctl.all_actions)):
            return False

        if self._has_vacancies_actions():
            return False

        return True

    def splitting_data(self) -> ProposalSplittingData:
        if self._splitting_data is None:
            self._splitting_data = ProposalSplittingData()

            persons_tickets = self._proposal_persons_tickets()
            for ticket in self._not_approved_tickets():
                self._splitting_data.persons.append(persons_tickets[ticket['key']])

            self._splitting_data.departments = self._department_being_deleted_with_unapproved_persons(
                unapproved_persons=self._splitting_data.persons,
            )

        return self._splitting_data

    def _not_approved_tickets(self) -> List[Any]:
        return [
            ticket
            for ticket in self._get_tickets_from_st(list(self._proposal_persons_tickets().keys()))
            if ticket[self.approval_field] != self.approval_field_value
        ]

    def _proposal_persons_tickets(self) -> Dict[str, str]:
        return {
            self._proposal_ctl.get_ticket_by_person_action(personal_action): personal_action['login']
            for personal_action in self._ticket_dispatcher.personal
            if self._proposal_ctl.get_ticket_by_person_action(personal_action) is not None
        }

    def _get_tickets_from_st(self, tickets: List[str]) -> Iterable[Any]:
        for i in range(0, len(tickets), MAX_TICKETS_PER_ST_REQUEST):
            yield from get_issues(keys=tickets[i:i+MAX_TICKETS_PER_ST_REQUEST])
