from typing import Optional

from django.db.models import Q

from staff.departments.vacancy_permissions import VacancyPermissions
from staff.departments.models import Vacancy
from staff.lib.utils.qs_values import localize, repack_related
from staff.person.models import Staff

from staff.proposal.controllers.department import order_field
from staff.proposal.controllers.utils import raw_data


class VacancyDataCtl(object):
    vacancy_fields = (
        'id',
        'name',
        'ticket',
        'offer_id',
        'headcount_position_code',

        'preprofile_id',
        'candidate_id',
        'candidate_first_name',
        'candidate_last_name',
        'candidate_login',
        'is_active',
        'status',
        'application_id',
        'department_id',
        'department__url',
        'department__level',
        'department__tree_id',
        'department__lft',
        'department__name',
        'department__name_en',

        'is_published',
    )

    def __init__(self, vacancy_ids, observer: Optional[Staff]):
        self._meta_data = {}

        # Ругаемся что вакансии не существует, если кто-то без прав доступа к вакансии пытается
        # либо добавить ее к существующей заявке либо добавить к новой заявке
        # условие not observer - VacancyDataCtl используется в VacancyGridFieldWithMeta
        filter_ = Q() if not observer else VacancyPermissions(observer).vacancy_filter_qs()

        vacancy_data_qs = (
            Vacancy.objects
            .filter(filter_)
            .filter(id__in=vacancy_ids, budget_position__isnull=False)
            .values(*self.vacancy_fields)
            .order_by('id')
        )
        self.vacancies_data = {
            v['id']: repack_related(v)
            for v in vacancy_data_qs
        }

    def as_form_data(self, vacancy_id):
        from staff.proposal.forms.vacancy import VacancyEditForm
        if not self.vacancies_data or vacancy_id not in self.vacancies_data:
            return {}

        result = raw_data(VacancyEditForm().data_as_dict())
        result['vacancy_id'] = vacancy_id
        return result

    def as_meta(self, vacancy_id):
        if not self.vacancies_data or vacancy_id not in self.vacancies_data:
            return {}

        meta = localize(self.vacancies_data[vacancy_id].copy())

        meta['vacancy_id'] = meta.pop('id')
        meta['department_url'] = meta['department']['url']
        meta['department_level'] = meta['department']['level']
        tree_id = meta['department']['tree_id']
        lft = meta['department']['lft']
        meta['department_order_field'] = order_field(tree_id, lft) if tree_id and lft else 0
        localize(meta['department'])
        meta['department_name'] = meta['department']['name']
        meta.pop('department')

        return meta
