from datetime import date, datetime
import random
from typing import Union, Dict, Any, Optional

from staff.departments.models import Department, DepartmentKind, DEPARTMENT_CATEGORY
from staff.lib.utils.date import parse_datetime
from staff.person.models import Staff


def _to_url(pk: Optional[int], can_be_missed: bool) -> Optional[str]:
    if not pk:
        return pk

    values_list = Department.objects.filter(id=pk).values_list('url', flat=True)

    if can_be_missed:
        return values_list.first()
    return values_list.get()


def _to_login(pk):
    if not pk:
        return pk
    return Staff.objects.filter(id=pk).values_list('login', flat=True).get()


def _to_code(pk):
    if not pk:
        return pk
    return DepartmentKind.objects.filter(id=pk).values_list('code', flat=True).get()


def _convert_url(action: Dict) -> Dict[str, Union[int, str, None]]:
    if action['url']:
        dep = Department.objects.get(url=action['url'])
        return {'id': dep.id, 'fake_id': None}

    return {'id': None, 'fake_id': action['fake_id']}


def _convert_name_from_form_data(action_name_section: Dict) -> Dict[str, Union[str, bool]]:
    return {
        'name': action_name_section['name'],
        'name_en': action_name_section['name_en'],
        'hr_type': action_name_section.get('hr_type', True),
        'is_correction': action_name_section.get('is_correction', False)
    }


def _convert_administration_from_form_data(action_administration_section: Dict) -> Dict[str, Any]:
    logins = [action_administration_section['chief']] + action_administration_section['deputies']
    ppl = dict(Staff.objects.filter(login__in=logins).values_list('login', 'id'))
    return {
        'chief': ppl.get(action_administration_section['chief']),
        'deputies': [ppl.get(l) for l in action_administration_section['deputies']],
    }


def _convert_hierarchy_from_form_data(action_hierarchy_section: Dict) -> Dict[str, Any]:
    parent_id = (
        Department.objects.values_list('id', flat=True).get(url=action_hierarchy_section['parent'])
        if action_hierarchy_section['parent']
        else None
    )

    return {
        'parent': parent_id,
        'fake_parent': action_hierarchy_section['fake_parent'],
        'changing_duties': action_hierarchy_section['changing_duties'],
    }


def _convert_technical_from_form_data(technical_action_section: Dict) -> Dict[str, Any]:
    code = {'code': technical_action_section['code']} if 'code' in technical_action_section else {}

    allowed_overdraft_percents = (
        {'allowed_overdraft_percents': technical_action_section['allowed_overdraft_percents']}
        if 'allowed_overdraft_percents' in technical_action_section
        else {}
    )

    return {
        'position': technical_action_section['order'],
        'kind': technical_action_section['department_type'],
        'category': technical_action_section['category'],
        **code,
        **allowed_overdraft_percents,
    }


def _convert_from_form_data(new_cleaned_data):
    converted_actions = []

    for action in new_cleaned_data['departments']['actions']:
        old_action = {'delete': action['delete']}
        old_action.update(_convert_url(action))

        if 'name' in action:
            old_action['name'] = _convert_name_from_form_data(action['name'])

        if 'administration' in action:
            old_action['administration'] = _convert_administration_from_form_data(action['administration'])

        if 'hierarchy' in action:
            old_action['hierarchy'] = _convert_hierarchy_from_form_data(action['hierarchy'])

        if 'technical' in action:
            old_action['technical'] = _convert_technical_from_form_data(action['technical'])

        converted_actions.append(old_action)

    return {
        'tickets': {
            'department_linked_ticket': new_cleaned_data.get('link_to_ticket', ''),
        },
        'apply_at_hr': '2018-05-02',  # 2do: генерируем или в новую форму добавляем 😣
        'apply_at': new_cleaned_data.get('apply_at', date.today()),
        'description': new_cleaned_data['description'],
        'actions': converted_actions,
        'persons': new_cleaned_data['persons'],
        'vacancies': new_cleaned_data['vacancies'],
        'headcounts': new_cleaned_data['headcounts'],
    }


def _convert_to_new_initial(old_initial):
    applied_at = old_initial.get('applied_at')
    apply_at = old_initial.get('apply_at')
    if apply_at:
        if isinstance(apply_at, date):
            apply_at = datetime.combine(apply_at, datetime.min.time())
        elif isinstance(apply_at, str):  # для случая с черновиком
            apply_at = parse_datetime(apply_at)

    initial_dict = {
        'applied_at': applied_at,
        'apply_at': apply_at or None,
        'description': old_initial.get('description', ''),
        'followers': [],
        'departments': {
            'link_to_ticket': old_initial.get('tickets', {}).get('department_linked_ticket', ''),
            'actions': [],
        },
        'persons': old_initial.get('persons', {'actions': []}),
        'vacancies': old_initial.get('vacancies', {'actions': []}),
        'headcounts': old_initial.get('headcounts', {'actions': []}),
        'splitted_from': old_initial.get('splitted_from'),
        'splitted_to': old_initial.get('splitted_to'),
    }

    for old in old_initial['actions']:
        new_action = {
            'url': _to_url(old['id'], old['fake_id'] is not None),
            'fake_id': old['fake_id'],
            'action_id': 'act_' + str(random.randrange(10000, 99999)),
            'sections': old['sections'],
            'delete': old['delete'],
        }

        _fix_fake_departments(initial_dict, new_action, old)

        if 'name' in old:
            new_action['name'] = {
                'name': old['name']['name'],
                'name_en': old['name']['name_en'],
                'hr_type': old['name'].get('hr_type'),
                'is_correction': old['name'].get('is_correction', False)
            }
        if 'hierarchy' in old:
            new_action['hierarchy'] = {
                'changing_duties': old['hierarchy'].get('changing_duties'),
                'parent': _to_url(old['hierarchy']['parent'], old['hierarchy']['fake_parent'] is not None),
                'fake_parent': old['hierarchy']['fake_parent'],
            }
        if 'administration' in old:
            new_action['administration'] = {
                'chief': _to_login(old['administration']['chief']),
                'deputies': [_to_login(pk) for pk in old['administration']['deputies']],
                'hr_partners': [],
                'budget_holder': None,
            }
        if 'technical' in old:
            new_action['technical'] = {
                'department_type': old['technical']['kind'] if 'kind' in old['technical'] else None,
                'code': old['technical']['code'] if 'code' in old['technical'] else _to_code(old['id']),
                'order': old['technical']['position'],
                'category':  old['technical'].get('category', DEPARTMENT_CATEGORY.NONTECHNICAL),
                'allowed_overdraft_percents': old['technical'].get('allowed_overdraft_percents', None),
            }

        initial_dict['departments']['actions'].append(new_action)

    return initial_dict


def _fix_fake_departments(initial_dict, new_action, old):
    if old['id'] is not None and new_action['url'] is None:
        for section in ('persons', 'vacancies', 'headcounts'):
            for action in initial_dict[section]['actions']:
                if 'department' in action and action['department'].get('fake_department') == old['fake_id']:
                    action['department']['department'] = None


def get_initial_old_style(proposal_ctl):
    initial_data = proposal_ctl.as_form_data()
    meta = proposal_ctl.get_meta()

    for action in initial_data['actions']:
        actions_metadata = meta.get(action['id'] or action['fake_id'], {})
        action['sections'] = actions_metadata.get('actions', [])
    return initial_data
